%-------------------------------------------------------------------------%
% Background
% Convert the mesh file from Gmsh to an appropriate format

% Notes
% 2D case
% quadrilateral element

% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

clear, clc
keyM = 1;   % 1: bare, 2: NM

fileComsol = ['Meshes/meshComsol' num2str(keyM) '.txt']; 
fileMesh = ['Meshes/meshUnitcell' num2str(keyM) '.txt']; 


%% Read mesh data

fgmsh = fopen(fileComsol,'r');
nStrings = 0;         % mark the names, 1-MeshFormat, 3-Nodes, 5-Elements
coords = [];
elems = [];
parts = [];

while ~feof(fgmsh)
    
    % read one line 
    tline = fgetl(fgmsh);
    ltline = length(tline);
    numtline = str2num(tline);
    
    % get the names
    if isempty(numtline) && ~isempty(tline)
        
        % detect the nodes
        if strcmpi(tline,'# Mesh point coordinates')
            nStrings = nStrings+1;
        end

        % detect the elements
        if strcmpi(tline,'# Elements')
            nStrings = nStrings+1;
        end
        
        % detect the parts
        if strcmpi(tline,'# Geometric entity indices')
            nStrings = nStrings+1;
        end
        
    end   
     
     % scan the data
     if ~isempty(numtline)  
            
         % read the nodes
         if nStrings == 1
                
             coords = [coords; roundn(numtline*100,-4)];   % normalization
         end
            
         % read the elements
         if nStrings == 2
             elems = [elems; numtline+1];
         end
         
         % read the part indexes
         if nStrings == 3
             parts = [parts; round(numtline/2)];
         end
            
     end
        
end

fclose(fgmsh);


%% Coordinate correction

cenX = (max(coords(:,1))+min(coords(:,1))) / 2;
cenY = (max(coords(:,2))+min(coords(:,2))) / 2;

coords(:,1) = coords(:,1) - cenX;
coords(:,2) = coords(:,2) - cenY;


%% Transverse boundary nodes

leftNodes = find(coords(:,1)==-0.5);
[~,index] = sort(coords(leftNodes,2));
leftNodes = leftNodes(index);

rightNodes = find(coords(:,1)==0.5);
[~,index] = sort(coords(rightNodes,2));
rightNodes = rightNodes(index);


%% Corner nodes

cornerNodes = [leftNodes(1) rightNodes(1) rightNodes(end) leftNodes(end)];


%% Plot the matrix configuration

nNodes = size(coords,1);
nElems = size(elems,1);

X0 = min(coords(:,1));
Y0 = min(coords(:,2));
    
L = max(coords(:,1)) - X0;
H = max(coords(:,2)) - Y0;
    
figure
hold on

% mesh
for iElem =1:nElems
     elemNodes = elems(iElem,:);
     elemNodes([3,4]) = elemNodes([4,3]);  % order correction

     if ispolycw(coords(elemNodes,1),coords(elemNodes,2))  % make sure nodes are numbered couterclockwise
         elemNodes = fliplr(elemNodes);
         
     end
     elems(iElem,:) = elemNodes;
     
     nodeZone = [elemNodes elemNodes(1)];
     
     % identify the part
     switch parts(iElem)
         case 1
            Col = [0.25 0.25 0.25];
         case 2
             Col = 'b';  
         case 3
             Col = 'r';
     end
     fill(coords(nodeZone,1),coords(nodeZone,2),Col,'edgecolor','k','linewidth',2);
     
end

% transverse boundary nodes
plot(coords(leftNodes,1),coords(leftNodes,2),'-','color','c','linewidth',10);
plot(coords(rightNodes,1),coords(rightNodes,2),'-','color','c','linewidth',10);

% corner nodes
plot(coords(cornerNodes,1),coords(cornerNodes,2),'o','color',[0.5 0.5 0.5],'linewidth',10,'MarkerEdgeColor','none','MarkerFaceColor','b','MarkerSize',20);
    
hold off
xlim([X0 X0+L]);
ylim([Y0 Y0+H]);
axis('equal')
axis off
set(gca,'xtick',[])
set(gca,'ytick',[])
box on
    
title('Mesh','FontName','Times New Roman','FontWeight','Bold','FontSize',16,'interpreter','latex')
set(gcf,'Units','centimeters','Position',[5 5 20 15]);

%% Create a mesh file

fmesh = fopen(fileMesh,'wt');

% nodes
fprintf(fmesh,'NNODES');
fprintf(fmesh,' %i\n',nNodes);

for iNode =1:nNodes
    fprintf(fmesh,'%i %f %f \n',iNode,coords(iNode,1),coords(iNode,2));
    
end
fprintf(fmesh,' \n');

% elems (attached by the part indexes)
fprintf(fmesh,'NELEMS');
fprintf(fmesh,' %i \n',nElems);

for iElem =1:nElems  
    fprintf(fmesh,'%i %i %i %i %i %i \n',iElem,elems(iElem,1),elems(iElem,2),elems(iElem,3),elems(iElem,4),parts(iElem));
    
end
fprintf(fmesh,' \n');

% transverse boundary nodes

fprintf(fmesh,'LEFT\n');
for i = 1:length(leftNodes)
    fprintf(fmesh,'%i ',leftNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');

fprintf(fmesh,'RIGHT\n');
for i = 1:length(rightNodes)
    fprintf(fmesh,'%i ',rightNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');

% corner nodes
fprintf(fmesh,'CORNERS\n');
for i = 1:length(cornerNodes)
    fprintf(fmesh,'%i ',cornerNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');


fclose(fmesh);
