%-------------------------------------------------------------------------%
% Background
% Main function for the macroscopic transient analysis on a finite
% LRAM panel fabricated by both normal and metamaterials
% Single-frequency excitation is considered

% Notes
% Same settings as those used in the frequency response analysis (read the stored data)
% Method: Newmark method

% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

clear,clc

%% Load the data from the frequency response analysis

keyCell = 2;              % 1: host, 2: NM

switch keyCell
    case 1
    load Macro_analysis/Data_fr_host
    selPeriod = 2;                       % selected time point in the following to investigate the field, 2 for host, 4 for NM 
    case 2
    load Macro_analysis/Data_fr_NM       
    selPeriod = 4;
end

% selFreq = [2000; 2500; 2000; 2500];   % for the metamaterial case

%% settings

% integration scheme
alpha = 0.2;
gamma = 1/2 + alpha;
beta = (1+alpha)^2 / 4;
rhoInf = 2/(1+alpha) - 1;            % used in COMSOL, generalized-alpha scheme


%% Transient analysis

keyTR = 2;                           % 1: in-plane, 2: out-of-plane
sampleTR = sampleFR;
keyU = 0;                            % 0: no plot

% initialize
nSelFreqs = length(selFreq);
TR = cell(1,nSelFreqs);
TRDNS = cell(1,nSelFreqs);

for iSelFreq = 1:length(selFreq)

    freq = selFreq(iSelFreq);

    TR{iSelFreq} = funTR(beta,gamma,freq,aUPres,uAmp,K,M,cpDofs,totDofs,presDofs,freeDofs,elemXID,numNd);
    
    % plot results under a single frequency
    [TR{iSelFreq}, TRDNS{iSelFreq}] = funTRField(TR{iSelFreq},namePanel,uAmp,keyTR,selCell,nCell,numL,sampleTR,selPeriod,keyU,sizeU);

end


%% Plot results together

% transient displacement response
nSelCells = length(selCell);
sizeUR = sizeU;
sizeUR(4) = sizeU(4)*nSelFreqs;

for iSel = 1:nSelCells
    figure
    set(gcf,'Units','centimeters','Position',sizeUR);
%     set(gcf,'Units','centimeters','Position',[5 0 15 30]);
    
    for iSelFreq = 1:nSelFreqs
        subplot(nSelFreqs,1,iSelFreq);
        box on
        hold on
        h1 = plot(TRDNS{iSelFreq}.T*TRDNS{iSelFreq}.freq,TRDNS{iSelFreq}.TD(iSel,:)/uAmp,'b-','linewidth',3);
        h2 = plot(TR{iSelFreq}.T*TR{iSelFreq}.freq,TR{iSelFreq}.TD(iSel,:)/uAmp,'r:','linewidth',3);

        ylabel('$u/U_{\rm{in}}$ [-]','FontName','Times New Roman','FontSize',20,'interpreter','latex');
        set(gca,'FontName','Times New Roman','FontSize',20,'XMinorTick','on','YMinorTick','on')

        if iSelFreq == 1
            lg = legend([h1 h2],'DNS','HM');
            set(lg,'FontSize',20,'interpreter','latex')
            legend boxoff  

        end

        if iSelFreq == nSelFreqs
            xlabel('$t/T$ [-]', 'FontName','Times New Roman','FontSize',20,'interpreter','latex')

        end

    end

end

% transient displacement field
figure
set(gcf,'Units','centimeters','Position',sizeUR);
hold on
box on
    
for iSelFreq = 1:nSelFreqs
    
    subplot(nSelFreqs,1,iSelFreq);
    box on
    hold on
    h1 = plot(TRDNS{iSelFreq}.XID/numL,TRDNS{iSelFreq}.selUt/uAmp,'b-','linewidth',3);
    h2 = plot(TR{iSelFreq}.XID(1:sampleTR:end)/numL,TR{iSelFreq}.selUt(1:sampleTR:end,:)/uAmp,'ro','linewidth',3,...
        'MarkerEdgeColor','none','MarkerFaceColor','r','MarkerSize',9);

    ylabel('$u/U_{\rm{in}}$ [-]','FontName','Times New Roman','FontSize',20,'interpreter','latex');
    set(gca,'FontName','Times New Roman','FontSize',20,'XMinorTick','on','YMinorTick','on')
    
    if iSelFreq == 1
        lg = legend([h1 h2],'DNS','HM');
        set(lg,'FontSize',20,'interpreter','latex')
        legend boxoff  
        
    end
    
    if iSelFreq == nSelFreqs
        xlabel('$\xi^{\rm{d}}/L_{\rm{tot}}$ [-]', 'FontName','Times New Roman','FontSize',20,'interpreter','latex') 

    end
    
end


%% Store the data
switch keyCell
    case 1
        save Data_tr_host alpha gamma beta rhoInf selPeriod keyTR sampleTR keyU TR TRDNS
    case 2
        save Data_tr_NM   alpha gamma beta rhoInf selPeriod keyTR sampleTR keyU TR TRDNS

end





