%-------------------------------------------------------------------------%
% Background
% Function to get effective properties using homogenization
% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

function HM = getHM(matProp,meshFile,W,Nset)

%-------------------------------------------------------------------------%
% compute the consituent elasticity
for i = 1:length(matProp)
Kc = matProp{i}.E/3/(1-2*matProp{i}.mu);
Gc = matProp{i}.E/2/(1+matProp{i}.mu);
matProp{i}.C = planeStrain(Kc,Gc);               % tangent elasticity

end

%-------------------------------------------------------------------------%
% read the unit cell mesh
[coords,elems,parts,boundNodes,cornerNodes] = readMesh(meshFile);

coords = coords * W;                 % scaling
nElems = length(elems);              % total element number

presNodes = cornerNodes;             % all corner nodes are prescribed
presDofs = getDofs(presNodes);       % prescribed DOFs       
presDof = length(presDofs);

Ar = W * 1;                   % referenc elamina area

%-------------------------------------------------------------------------%
% select integration scheme
coordsElem = coords(elems(1,:),:);     % element sample
order = 0;                             % default value for the quadrilateral element

% get element properties
elemProp = getElemProp(coordsElem,order);

%-------------------------------------------------------------------------%
% assemble system stiffness and mass matrice
[K,M] = assembleKeMe(coords,elems,parts,matProp,elemProp,nElems,order);

%-------------------------------------------------------------------------%
% partition K and M as [ii,id;di,dd] 
[indNodes,depNodes,freeNodes,Tdi] = relationInd(coords, presNodes, boundNodes);

indDofs = getDofs(indNodes);              % basic independent DOFs
indDof = length(indDofs);                 % number
depDofs = getDofs(depNodes); 
freeDofs =  getDofs(freeNodes);

Kii = K(indDofs,indDofs);
Kid = K(indDofs,depDofs);
Kdi = K(depDofs,indDofs);
Kdd = K(depDofs,depDofs);

Mii = M(indDofs,indDofs);
Mid = M(indDofs,depDofs);
Mdi = M(depDofs,indDofs);
Mdd = M(depDofs,depDofs);

%-------------------------------------------------------------------------%
% remove dependent nodes (KEii*ui + MEii*ai = fiEii)
KEii = Kii +Kid *Tdi + Tdi'*Kdi +Tdi'*(Kdd*Tdi);
MEii = Mii +Mid *Tdi + Tdi'*Mdi +Tdi'*(Mdd*Tdi);

% partition KEii and MEii as[pp,pf;fp,ff]
indIndices  = 1:indDof;
presIndices = [1 2 3 5 7];
freeIndices = setdiff(indIndices, presIndices);

Kpp = KEii(presIndices,presIndices);
Kpf = KEii(presIndices,freeIndices);
Kfp = KEii(freeIndices,presIndices);
Kff = KEii(freeIndices,freeIndices);

Mpp = MEii(presIndices,presIndices);
Mpf = MEii(presIndices,freeIndices);
Mfp = MEii(freeIndices,presIndices);
Mff = MEii(freeIndices,freeIndices);
%-------------------------------------------------------------------------%
% craig-Bamptom mode synthesis technique

th1 = clock;
% compute quasistatic response
% get static condensate
S = -inv(Kff) * Kfp;

% get Kqs (i.e.KM) and Mqs (i.e.MM)
Kqs = Kpp + Kpf*S;
Mqs = S'*Mff*S + 2*Mpf*S + Mpp;

% compute internal dynamics
[Fai,Omega2] = eigs(Kff,Mff,Nset.nModes,'sm');
dispM = sqrt(real(Fai'*Mff*Fai));                     % modal displacement

% normalize the modal displacement
for i = 1:length(dispM)
    Fai(:,i) = Fai(:,i)/dispM(i,i); 
end

% sort (low to high frequency)
[Omega,index] = sort(real(sqrt(diag(Omega2)))); 
Fai = Fai(:,index);

HM.resFreq = Omega/2/pi;             % Hz
HM.Fai = Fai;

th2 = clock;
HM.dth = etime(th2,th1);                  % computation time for mode decomposition

%-------------------------------------------------------------------------%
% get quasi-static effective elasticity, inertia and mass density
presCoords = zeros(2*length(presNodes),1);
for i = 1:2
   presCoords(i:2:end) = coords(presNodes,i);
end

% get in-plane and out-of-plane indices
hatIndices = [1 3 4 5];
tildeIndices = [2];
indices{1} = hatIndices;
indices{2} = tildeIndices;

Ar = W*1;                                          % reference surface area

th1 = clock;
[CM, DM, rhoM] = getCDRho(presCoords,indices,presIndices,Kqs,Mqs,Ar);  % CM and DM are completely in-plane
HM.CM = CM;
HM.DM = DM;
HM.rhoM = rhoM;

% get coupling coefficients
HM = getHJ(presCoords,indices,presIndices,Nset,Mff,Mpf,S,HM);          % Hc is completely in-plane
th2 = clock;

HM.dth = HM.dth + etime(th2,th1);                  % add the time to compute effective properties

% compute JJll/Ar, HHll/Ar and HHrr/Ar
HM.JJ.ll = [HM.Jc.lm(:,1).*HM.Jc.lm(:,1) HM.Jc.lm(:,2).*HM.Jc.lm(:,2) ...
            HM.Jc.lm(:,1).*HM.Jc.lm(:,2) HM.Jc.lm(:,2).*HM.Jc.lm(:,1)] / Ar;   % 2*2: 11, 22, 12, 21
        
HM.HH.ll = conj(HM.Hc.lm).*HM.Hc.lm / Ar;            % 1*1
HM.HH.rr = conj(HM.Hc.rt).*HM.Hc.rt / Ar;   

end


%-------------------------------------------------------------------------%
% Function to compute quasi-static effective properties
% CM: mm, bb,1*1
% DM: mm,bb, 1*1
% rhoM: ll, rr, 2*2
%-------------------------------------------------------------------------%

function [CM, DM, rhoM] = getCDRho(presCoords,indices,presIndices,Kqs,Mqs,Ar)

% get in-plane projections of Kqs and Mqs
hatKqs = Kqs(indices{1},indices{1});
hatMqs = Mqs(indices{1},indices{1});

% compute in-plane quasi-static effective elasticity
% presCoords(2:2:end) is the same as Eta
CM.mm = (presCoords(1:2:end)'*hatKqs*presCoords(1:2:end))'/Ar;
CM.bb = ((presCoords(2:2:end).*presCoords(1:2:end))'*hatKqs*(presCoords(2:2:end).*presCoords(1:2:end)))'/Ar;

% compute in-plane quasi-static effective elasticity
DM.ll = (presCoords(1:2:end)'*hatMqs*presCoords(1:2:end))'/Ar;
DM.rr = ((presCoords(2:2:end).*presCoords(1:2:end))'*hatMqs*(presCoords(2:2:end).*presCoords(1:2:end)))'/Ar;

% compute quasi-static effective mass density
presIndice = length(presIndices);
Ip = ones(presIndice,1);

tildePresCoords = [presCoords(2:2:end);presCoords(2:2:end);];
tildePresCoords = tildePresCoords(presIndices);

for i = 1:2
    for j = 1:2
    rhoM.ll(i,j) = (Ip(indices{i})'*Mqs(indices{i},indices{j})*Ip(indices{j}))/Ar;    
    rhoM.rr(i,j) = ((tildePresCoords(indices{i}).*Ip(indices{i}))'*Mqs(indices{i},indices{j})*(tildePresCoords(indices{j}).*Ip(indices{j})))/Ar; 
        
    end
end
    
end


%-------------------------------------------------------------------------%
% Function to compute coupling coefficients
%-------------------------------------------------------------------------%
function HM = getHJ(presCoords,indices,presIndices,Nset,Mff,Mpf,S,HM)
Jc.lm = zeros(Nset.nModes,2);      % 1*2
Jc.rt = zeros(Nset.nModes,2);      % 1*2

Hc.lm = zeros(Nset.nModes,1);      % 1*1
Hc.rt = zeros(Nset.nModes,1);      % 1*1

presIndice = length(presIndices);
Ip = ones(presIndice,1);

tildePresCoords = [presCoords(2:2:end);presCoords(2:2:end);];
tildePresCoords = tildePresCoords(presIndices);

for iMode = 1:Nset.nModes
    
    % current mode
    fai = HM.Fai(:,iMode);
    
    % compute the coupling relation vector
    MC = S'*Mff*fai + eye(presIndice)*Mpf*fai;
    hatMC = MC(indices{1});           % in-plane projection
    
    % compute jc components
    lm = zeros(1,2);
    rt = zeros(1,2);
    for i = 1:2
         lm(i) = Ip(indices{i})'* MC(indices{i});
         rt(i) =  (tildePresCoords(indices{i}).*Ip(indices{i}))'* MC(indices{i});
    end
    
    Jc.lm(iMode,:) = lm;
    Jc.rt(iMode,:) = rt;
    
    % compute hc components
    Hc.lm(iMode,:) = hatMC'*presCoords(1:2:end);
	Hc.rt(iMode,:) = hatMC'*(presCoords(2:2:end).*presCoords(1:2:end));
    
end

% minimize HM
idActive = find(HM.resFreq<Nset.sf*Nset.freqLim); % only low-order localized modes are activated

HM.resFreq = HM.resFreq(idActive,:);
HM.Fai = HM.Fai(:,idActive);

HM.Jc.lm = Jc.lm(idActive,:);
HM.Jc.rt = Jc.rt(idActive,:);

HM.Hc.lm = Hc.lm(idActive,:);
HM.Hc.rt = Hc.rt(idActive,:);


end