%-------------------------------------------------------------------------%
% Background
% Main function for the macroscopic frequency response analysis on a finite
% LRAM panel fabricated by both normal and metamaterials
% The flexural wave is focused on (almost invisiable body wave under the 
% low frequency condition)

% Notes
% State: plane strain 
% Behaviour: linear elasticity
% Boundary: prescribed displacements on the ends (displacement control),
% no external nodal force and moment resultants on the free 
% control points (more seriously speaking, freee Dofs)
% Method: isogeometry analysis, univariate B-spline basis functions, open 
% knot vector
% Unit system (unit thickness): N,Pa,m,kg;
% Smart choice is made on the control points to ahieve a linear coordinate
% mapping so that the integration can be directly performed on the 
% physical domain. Otherwise, on the parameter domain.

% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

clear,clc


%% Homogenization settings

% governing parameters to select the principal modes
Nset.nModes = 9;                       % maximum mode number
Nset.freqLim = 2e3;                   % maximum frequency, reference 2e3
Nset.sf = 1;                          % safety factor


%% Input constituent material parameters

% Glass matrix (part 1)
matrixProp.E = 107e9;                           % Young's modulus
matrixProp.mu = 0.276;                            % poisson ratio
matrixProp.rho = 2.05e3;                         % mass density

% rubber coating (part 2)
coatProp.E = 35e3;                          
coatProp.mu = 0.469;
coatProp.rho = 1.30e3;

% Tungsten core (part 3)
coreProp.E = 411e9;
coreProp.mu = 0.28;
coreProp.rho = 19.2e3;


%% Allocate constituent materials and geometries

% normal material, only matrix
matPropNM{1} = matrixProp;
meshFileNM = ['Meshes/meshUnitcell1.txt'];

% metamaterial
matPropMM = cell(1,3);
matPropMM{1} = matrixProp;
matPropMM{2} = coatProp;
matPropMM{3} = coreProp;
meshFileMM = ['Meshes/meshUnitcell2.txt'];

% unit cell representative size
W = 10e-3;                            % width, for scaling


%% Compute effective properties

matHM = cell(2,1);    % 1: normal material, 2: metamaterial

matHM{1} = getHM(matPropNM,meshFileNM,W,Nset);
matHM{2} = getHM(matPropMM,meshFileMM,W,Nset);

% for convenience, reassemble effective properties (remove inactive rotational modes)
matENR = cell(2,1);
modeSel = [2 3 6 5];               % in-plane, out-of-plane, in-plane, out-of-plane

for i = 1:2
    matENR{i}.CM = matHM{i}.CM;
    matENR{i}.DM = matHM{i}.DM;
    matENR{i}.rhoM = matHM{i}.rhoM;
    
    matENR{i}.resOmega = zeros(4,1);
    matENR{i}.Jc = zeros(4,1);     % lm_hat, lm_tilde, lm_hat, lm_tilde
    matENR{i}.Hc = zeros(4,1);     % lm, rt, lm, rt
    
end

matENR{2}.resOmega = matHM{2}.resFreq(modeSel)*2*pi;

matENR{2}.Jc([1 3]) = matHM{2}.Jc.lm(modeSel([1 3]),1);
matENR{2}.Jc([2 4]) = matHM{2}.Jc.lm(modeSel([2 4]),2);

matENR{2}.Hc([1 3]) = matHM{2}.Hc.lm(modeSel([1 3]));
matENR{2}.Hc([2 4]) = matHM{2}.Hc.rt(modeSel([2 4]));


%% Macroscopic analysis settings

% currently pure normal or metamaterial is considered
keyCell = 2;
nCell = 50;                     % number of unit cells

% number of unit cells
switch keyCell
    case 1
    nCellNM = nCell;                % left, normal material unit cells
    nCellMM = 0;                 % right, metamaterial unit cells
    namePanel = 'host';

    case 2
    nCellNM = 0;                  
    nCellMM = nCell;   
    namePanel = 'NM';            % negative mass
    
end

numL = W*nCell;               % total length
uAmp = 1e-3;                  % excitation linear displacement amplitude
alphaAmp = 0;                 % excitation rotational displacement amplitude
eC = 1;                       % out-of-plane vector basis

%-------------------------------------------------------------------------%
% define symbolic variables
syms xiP xiD L

% governing parameters
p = 3;                       % spline order
m = 16;                       % number of elements, estimated based on the minimal wavelength

mNodes = (1:m+1)';           % element nodes
mCoords = (linspace(0,1,m+1))'*L;   % element nodal coordinates
mElems = [mNodes(1:end-1) mNodes(2:end)]; % elements

n = m+p;                     % number of control points or basis functions  

% open knot vector (xiP, in the parameter domain)
% two open ends for convenience of prescribing linear displacements
knot = [ 0*ones(1,p) 0:m  m*ones(1,p) ]; % a series of xiP

% control point vector (xiD, in the physical domain)
cpNodes = (1:n)';              % control point indexes
cpDofs = getDofs(cpNodes);   % control point DOFs

cpXiD = linspace(0,1,m+1);
cpXiD = [cpXiD(1);
         cpXiD(2)/3;             % inserted two control points  
         cpXiD(2:end-1)';
         cpXiD(end)-cpXiD(2)/3
         cpXiD(end)]*L;

% construct the basis function pieces on the physical domain
% significant errors may appear on the interface between the normal and metamaterials
% due to same basis functions are applied on the both original and enriched
% DOFs
sizeNp = [5 0 30 14];
sizeNd = [5 0 30 24];
nxiP = 20+1;       % sampling point number in each element piece
keyN = 1;         % 0: no plot
[Nd, dNd, dNd2] = funBasis(knot,cpXiD,n,m,p,nxiP,keyN,sizeNp,sizeNd);

%-------------------------------------------------------------------------%

% boundary conditions
cpPresNodes = [1; m+3];                             % prescribed left and right ends
cpPresDofs = getDofs(cpPresNodes);

aUPres = [0; 1; 0; 0]*uAmp;                           % linear displacement amplitude
aAlphaPres = []*alphaAmp;                      % rotational displacement amplitude

% if ~isempty(aAlphaPres)
%     % convert the rotational displacement to the linear displacement
%     numdNd1 = double(subs(dNd(1,1),{xiD},0));             % on the left end
%     numdNd2 = double(subs(dNd(1,1),{xiD},0));
% 
%     aUPres(5) = (aAlphaPres - (-eC*aUPres(2)*numdNd1))/(-eC*numdNd2);
%     cpPresDofs(3) = 5; 
% end


%% Plot material distribution

rCell = nCellNM/nCell;
mFlags = 2*ones(m,1);                          % mark different material zones
mFlags(1:round(m*rCell),1) = 1;     

cpFlags = 2*ones(m+3,1);
cpFlags(1:round(m*rCell),1) = 1;               % only true for pure normal or metamaterial     

sizeNm = [5 0 20 5];           
figure
set(gcf,'Units','centimeters','Position',sizeNm);
axis off
hold on
for mm = 1:m
    mInternval = subs(mCoords(mElems(mm,:)),{L},1);

    switch mFlags(mm)
    case 1
    Col = [0.5 0.5 0.5];
    
    case 2
    Col = 'b';
    end
    plot(mInternval,[0 0],'-x','color',Col,'linewidth',6,'MarkerEdgeColor','k','MarkerFaceColor','k','MarkerSize',12);
end


%% Compute system stiffness and mass matrices

Ar = W*1;

tkm1 = clock;
[K,M] = compKM(Nd,dNd,dNd2,mCoords,mElems,mFlags,cpDofs,matENR,numL,Ar,eC,keyCell);

tkm2 = clock;
FR.dtkm = etime(tkm2,tkm1);                                                % computation time for computing systematic matrices


%% Partition preparation

cpDof = length(cpDofs);
switch keyCell
    case 1
    totDofs = cpDofs;
    case 2
    % take the enriched DOFs into account
    totDofs = [cpDofs; cpDofs+cpDof; cpDofs+2*cpDof];    
end
presDofs = cpPresDofs;
freeDofs = setdiff(totDofs,presDofs);


%% Eigenfrequency analysis (to check the convergence, only for the host case)

EA.nMode = 4;           % preset number of eigenmodes, according to the DNS (0-2000Hz)
                                 % accurate solution 119.86, 477.17, 1065.4, 1873.9Hz

if keyCell == 1
EA = compEA(EA,K,M,freeDofs);
EA.freq
end


%% Frequency response analysis 

% generate element pieces on the physical domain
elemXID = zeros(m,nxiP); 

for i = 1:m
    elemXID(i,:) = linspace(i-1,i,nxiP);

end
elemXID = elemXID*numL/m;   

FR.freq = 0:5:2500;                                              % 0-2500 for the metamaterial case, to confirm 
                                                                           % from where the homogenization method does not work well
[FR, numNd] = compFR(FR,Nd,K,M,cpDofs,totDofs,presDofs,freeDofs,aUPres,elemXID,numL);


%% Plot the frequency response functions versus those from DNS

% compute the frequency response function of interest
% interpolate the displacement amplitude on the interested position
selCell = [20 40];
keyFR = 2;                          % 1: in-plane, 2: out-of-plane
sizeFR =  [5 0 10 15];

[FR, FRDNS] = funFRF(namePanel,nCell,numL,FR,selCell,keyFR,uAmp,sizeFR);


%% plot the steay-sate displacement fields of interest

switch keyCell
    case 1
        selFreq = [150; 400; 900; 1650];             % selected frequency
    case 2
        selFreq = [250; 500; 1000; 1750];           % 2000, 2500Hz as two extra cases for the metamaterial case

end

sizeU = [5 0 20 7.5]; 
sampleFR = m;

[FR, FRDNS] = funFRField(namePanel,FR,FRDNS,numL,selFreq,keyFR,uAmp,sampleFR,sizeU);


%% Store the data
switch keyCell
    case 1
        save Data_fr_host
    case 2
        save Data_fr_NM

end




