%-------------------------------------------------------------------------%
% Background
% Function to get system stiffness matrix and mass matrix (take both elastic 
% inertia and mass density into account
% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

function [K,M] = compKM(Nd,dNd,dNd2,mCoords,mElems,mFlags,cpDofs,matENR,numL,Ar,eC,keyCell)

syms xiD L
n = length(cpDofs(1:2:end));

% initialize
% for the external dynamics 
Kee = cell(2,1);    % in-plane, out-of-plane
Mee = cell(2,1);

for i = 1:2
    Kee{i} = zeros(n,n);
    Mee{i} = zeros(n,n);
end

if  keyCell == 2 
    %-------------------------------------------------------------%
    % for the internal dynamics (if any)
    Kii = cell(4,1);    % in-plane, out-of-plane, in-plane, out-of-plane
    Mii = cell(4,1);

    % coupling
    Mcc = cell(4,1);

    for i = 1:4
    Kii{i} = zeros(n,n);
    Mii{i} = zeros(n,n);
    Mcc{i} = zeros(n,n);

    end
end

% define a set of normalized integrals
numintSet = zeros(3,3);
intSet = sym(numintSet);

for i = 1:n
    % identify nonzero elements of Ndi
    indexI = find(Nd(i,:)~=0);

    for j = 1:n
        % identify nonzero elements of Ndj
        indexJ = find(Nd(j,:)~=0);

        % pick the common nonzero elements
        indexIJ = intersect(indexI,indexJ);
        m1 = min(indexIJ);
        m2 = max(indexIJ);

        for im = m1:m2
            % integration interval
            mInternval = mCoords(mElems(im,:));
            
            % compute the normalized integrals to be used
            intSet(1,1) = int(Nd(i,im)*Nd(j,im),xiD,mInternval(1),mInternval(2));     
            intSet(2,2) = int(dNd(i,im)*dNd(j,im),xiD,mInternval(1),mInternval(2));
            intSet(3,3) = int(-eC*dNd2(i,im)*dNd2(j,im)*(-eC),xiD,mInternval(1),mInternval(2));
            
            intSet(2,1) = int(dNd(i,im)*Nd(j,im),xiD,mInternval(1),mInternval(2));     
            intSet(3,1) = int(-eC*dNd2(i,im)*Nd(j,im),xiD,mInternval(1),mInternval(2));     
            
            numintSet = double(subs(intSet,{L},{numL}));        % 1D case so the material property can be taken outside
            
            % identify the material
            mat = matENR{mFlags(im)};
            
            % body wave 
            Kee{1}(i,j) = Kee{1}(i,j) + numintSet(2,2)*mat.CM.mm;
            Mee{1}(i,j) = Mee{1}(i,j) + numintSet(2,2)*mat.DM.ll + numintSet(1,1)*mat.rhoM.ll(1,1);
            
            % flexural wave
            Kee{2}(i,j) = Kee{2}(i,j) + numintSet(3,3)*mat.CM.bb;
            Mee{2}(i,j) = Mee{2}(i,j) + numintSet(3,3)*mat.DM.rr + numintSet(1,1)*mat.rhoM.ll(2,2);
            
            if  keyCell == 2 
                %-------------------------------------------------------------%
                % internal dynamics
                % coupling 1
                Kii{1}(i,j) = Kii{1}(i,j) + numintSet(1,1) * (mat.resOmega(1))^2;
                Mii{1}(i,j) = Mii{1}(i,j) + numintSet(1,1) * 1;
                Mcc{1}(i,j) = Mcc{1}(i,j) + numintSet(2,1) * mat.Hc(1) + numintSet(1,1) * mat.Jc(1);

                % coupling 2
                Kii{2}(i,j) = Kii{2}(i,j) + numintSet(1,1) * (mat.resOmega(2))^2;
                Mii{2}(i,j) = Mii{2}(i,j) + numintSet(1,1) * 1;
                Mcc{2}(i,j) = Mcc{2}(i,j) + numintSet(3,1) * mat.Hc(2) + numintSet(1,1) * mat.Jc(2);

                % coupling 3
                Kii{3}(i,j) = Kii{3}(i,j) + numintSet(1,1) * (mat.resOmega(3))^2;
                Mii{3}(i,j) = Mii{3}(i,j) + numintSet(1,1) * 1;
                Mcc{3}(i,j) = Mcc{3}(i,j) + numintSet(2,1) * mat.Hc(3) + numintSet(1,1) * mat.Jc(3);

                % coupling 4
                Kii{4}(i,j) = Kii{4}(i,j) + numintSet(1,1) * (mat.resOmega(4))^2;
                Mii{4}(i,j) = Mii{4}(i,j) + numintSet(1,1) * 1;
                Mcc{4}(i,j) = Mcc{4}(i,j) + numintSet(3,1) * mat.Hc(4) + numintSet(1,1) * mat.Jc(4);
            end

        end
        
    end

end

% form the total stiffness and mass matrices

switch keyCell
    
    case 1
        K = zeros(2*n*1,2*n*1);   
        M = zeros(2*n*1,2*n*1);
        
        for i = 1:2
        K(i:2:end,i:2:end)  = Kee{i};
        M(i:2:end,i:2:end)  = Mee{i};
        end
        
    case 2
    Kp = cell(2,1);
    Mp = cell(2,1);

    K = zeros(2*n*3,2*n*3);   
    M = zeros(2*n*3,2*n*3);

    for i = 1:2
        Kp{i} = zeros(3*n,3*n);
        Mp{i} = zeros(3*n,3*n);

    end

    for i = 1:2

    Kp{i}(1:n,1:n) = Kee{i};      
    Kp{i}(n+1:2*n,n+1:2*n) = Kii{i};    
    Kp{i}(2*n+1:3*n,2*n+1:3*n) = Kii{i+2};  

    Mp{i}(1:n,1:n) = Mee{i};
    Mp{i}(n+1:2*n,n+1:2*n) = Mii{i};
    Mp{i}(2*n+1:3*n,2*n+1:3*n) = Mii{i+2};
    Mp{i}(1:n,n+1:3*n) = 1/Ar*[Mcc{i} Mcc{i+2}];
    Mp{i}(n+1:3*n,1:n) = [Mcc{i} Mcc{i+2}]';

    % total

    K(i:2:end,i:2:end) = Kp{i};
    M(i:2:end,i:2:end) = Mp{i};

    end

end

end