%-------------------------------------------------------------------------%
% Background
% Main function for the unit cell analysis (continuum, 3 constituents, symmetric)
% Thin LRAM panel with both body and flexural waves taken into account

% Notes
% State: plane strain 
% Behaviour: linear elasticity
% Boundary: lamina-level periodic boundary conditions
% Element type: quadrilateral element
% Unit system: N, Pa, m, kg, unit thickness
% Storage: sparse matrix

% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

clear,clc
warning off all


%% Settings

% material
matProp = cell(1,3);

% Glass (part 1)
matProp{1}.E = 107e9;                           % Young's modulus
matProp{1}.mu = 0.276;                            % poisson ratio
matProp{1}.rho = 2.05e3;                         % mass density

Kc = matProp{1}.E/3/(1-2*matProp{1}.mu);
Gc = matProp{1}.E/2/(1+matProp{1}.mu);
matProp{1}.C = planeStrain(Kc,Gc);               % tangent elasticity

% rubber coating (part 2)
matProp{2}.E = 35e3;                          
matProp{2}.mu = 0.469;
matProp{2}.rho = 1.30e3;

Kc = matProp{2}.E/3/(1-2*matProp{2}.mu);
Gc = matProp{2}.E/2/(1+matProp{2}.mu);
matProp{2}.C = planeStrain(Kc,Gc);      

% Tungsten core (part 3)
matProp{3}.E = 411e9;
matProp{3}.mu = 0.28;
matProp{3}.rho = 19.2e3;

Kc = matProp{3}.E/3/(1-2*matProp{3}.mu);
Gc = matProp{3}.E/2/(1+matProp{3}.mu);
matProp{3}.C = planeStrain(Kc,Gc);   

% geometry
W = 10e-3;                                % width

fOut = [0.5 0.4];                         % outer size
fIn1 = [0.25 0.2 0.15];                   % inner size 1
fIn2 = [0.125 0.1 0.075];                 % inner size 2

matProp{3}.V = (fIn2(1)*fIn2(3)+pi/4*fIn2(1)*(fIn2(2)-fIn2(3))) * 4*W^2 * 1;   % volume
matProp{3}.m = matProp{3}.rho * matProp{3}.V;                             % mass

matProp{2}.V = (fIn1(1)*fIn1(3)+pi/4*fIn1(1)*(fIn1(2)-fIn1(3))) * 4*W^2 * 1 - matProp{3}.V;
matProp{2}.m = matProp{2}.rho * matProp{2}.V;

matProp{1}.V = fOut(1)*fOut(2) * 4*W^2 * 1 - matProp{3}.V - matProp{2}.V;
matProp{1}.m = matProp{1}.rho * matProp{1}.V;

% mesh file
keyCell = 2;
meshFile = ['Meshes/meshUnitcell' num2str(keyCell) '.txt'];     % 1: host, 2: NM

% governing parameters
fImg = 0.25;                          % to scale the imaginary axis if necessary

Nset.dirWave = [1 0];                 % wave vector
Nset.nMode = 9;                       % maximum mode number
Nset.nB = 2;                          % branch number
Nset.nSample = 201;                   % for k sampling
Nset.nOmega = 25*(Nset.nSample-1)+1;  % for omega sampling
Nset.lambdaTh = 5*W;                  % wavelength limitation, actually should be larger than 6*H
Nset.freqLim = 2e3;                   % maximum frequency, reference 2e3
Nset.sf = 1;                          % safety factor
Nset.method = [0 1];                  % Bloch analysis, homogenization
                                      % 0: only k-omega form, 1: both k-omega and omega-k form

%% Read the mesh

[coords,elems,parts,boundNodes,cornerNodes] = readMesh(meshFile);

coords = coords * W;                 % scaling

nNode = length(coords);             % total node number
totDof = 2*nNode;                   % total DOFs

nElems = length(elems);              % total element number

presNodes = cornerNodes;             % all corner nodes are prescribed
presDofs = getDofs(presNodes);       % prescribed DOFs  
presDof = length(presDofs);

Ar = W * 1;                   % referenc elamina area

%% Select integration scheme

coordsElem = coords(elems(1,:),:);     % element sample
order = 0;                             % default value for the quadrilateral element

% get element properties
elemProp = getElemProp(coordsElem,order);
totInt = elemProp.nInts*nElems;              % total integration points


%% Assemble system stiffness and mass matrice

tkm1 = clock;
[K,M] = assembleKeMe(coords,elems,parts,matProp,elemProp,nElems,order);

tkm2 = clock;
HM.dtkm = etime(tkm2,tkm1);                                                % computation time for computing systematic matrices

figure
SizeM = [5 5 20 15];
set(gcf,'Units','centimeters','Position',SizeM);

hold on

matrixNodes = zeros(nNode,1);   % mark the matrix nodes

% plot the matrix
for iElem =1:nElems
     elemNodes = elems(iElem,:);
     nodeZone = [elemNodes elemNodes(1)];
     
     % identify the part
     switch parts(iElem)
         case 1
            Col = [0.5 0.5 0.5];
            matrixNodes(nodeZone) = 1;
            
         case 2
             Col = 'b';  
         case 3
             Col = 'r';
     end
     
     fill(coords(nodeZone,1),coords(nodeZone,2),Col,'edgecolor','k','linewidth',2);
        
end

axis equal
axis tight
axis off
set(gca,'xtick',[])
set(gca,'ytick',[])


%% Independent, dependent and free nodes

tp1 = clock;
[indNodes,depNodes,freeNodes,Tdi] = relationInd(coords, presNodes, boundNodes);

indDofs = getDofs(indNodes);              % basic independent DOFs
indDof = length(indDofs);                 % number

depDofs = getDofs(depNodes); 
depDof = length(depDofs);

freeDofs =  getDofs(freeNodes);
freeDof = length(freeDofs);


%% partition K and M as [ii,id;di,dd] 
Kii = K(indDofs,indDofs);
Kid = K(indDofs,depDofs);
Kdi = K(depDofs,indDofs);
Kdd = K(depDofs,depDofs);

Mii = M(indDofs,indDofs);
Mid = M(indDofs,depDofs);
Mdi = M(depDofs,indDofs);
Mdd = M(depDofs,depDofs);

% remove dependent nodes (KEii*ui + MEii*ai = fiEii)
KEii = Kii +Kid *Tdi + Tdi'*Kdi +Tdi'*(Kdd*Tdi);
MEii = Mii +Mid *Tdi + Tdi'*Mdi +Tdi'*(Mdd*Tdi);


%% Partition KEii and MEii as[pp,pf;fp,ff]

% relax the DOFs of corner nodes except one
indIndices  = 1:indDof;
presIndices = [1 2 3 5 7];
freeIndices = setdiff(indIndices, presIndices);

% partition KEii
Kpp = KEii(presIndices,presIndices);
Kpf = KEii(presIndices,freeIndices);
Kfp = KEii(freeIndices,presIndices);
Kff = KEii(freeIndices,freeIndices);

% partition MEii
Mpp = MEii(presIndices,presIndices);
Mpf = MEii(presIndices,freeIndices);
Mfp = MEii(freeIndices,presIndices);
Mff = MEii(freeIndices,freeIndices);

tp2 = clock;
HM.dtp = etime(tp2,tp1);                                                   % computation time for partition


%% Craig-Bamptom mode synthesis technique

th1 = clock;
% compute quasistatic response
% get static condensate
S = -inv(Kff) * Kfp;

% get Kqs (i.e.KM) and Mqs (i.e.MM)
Kqs = Kpp + Kpf*S;
Mqs = S'*Mff*S + 2*Mpf*S + Mpp;

% compute internal dynamics
[Fai,Omega2] = eigs(Kff,Mff,Nset.nMode,'sm');
dispM = sqrt(real(Fai'*Mff*Fai));                     % modal displacement

% normalize the modal displacement
for i = 1:length(dispM)
    Fai(:,i) = Fai(:,i)/dispM(i,i); 
end

% sort (low to high frequency)
[Omega,index] = sort(real(sqrt(diag(Omega2)))); 
Fai = Fai(:,index);

HM.resFreq = Omega/2/pi;             % Hz
HM.Fai = Fai;

th2 = clock;
HM.dth = etime(th2,th1);                                                   % computation time for mode decomposition

% recover the full eigen modal shapes
Fai_full = zeros(nNode,Nset.nMode);

for i = 1:length(dispM)
    Fai_full(indDofs(freeIndices),i) = Fai(:,i);
    Fai_full(depDofs,i)  = Tdi* Fai_full(indDofs,i);
    
end
HM.Fai_full = Fai_full;


%% Plot eigen modal displacement fields from model reduction
Ueig    = HM.Fai_full;
ModeSel = [1 2 3];          % choose mode 1 to show
keyArrow = 1;           % show displacement direction
keyAll = 'Inclusion';   % only the displacement direction of the inclusion
funPlotModes(coords,elems,Ueig,ModeSel,SizeM,keyArrow,keyAll,matrixNodes);


%% Computational homogenization

[HM, kHDA, omegaHDA] = compHom(coords,presNodes,presIndices,Kqs,Mqs,Mff,Mpf,S,HM,W,Nset);

% compute JJll/Ar, HHll/Ar and HHrr/Ar
HM.JJ.ll = [HM.Jc.lm(:,1).*HM.Jc.lm(:,1) HM.Jc.lm(:,2).*HM.Jc.lm(:,2) ...
            HM.Jc.lm(:,1).*HM.Jc.lm(:,2) HM.Jc.lm(:,2).*HM.Jc.lm(:,1)] / Ar;   % 2*2: 11, 22, 12, 21
        
HM.HH.ll = conj(HM.Hc.lm).*HM.Hc.lm / Ar;            % 1*1
HM.HH.rr = conj(HM.Hc.rt).*HM.Hc.rt / Ar;  


%% Bloch analysis

[kBA, omegaBA]= blochAnalysis(K,M,boundNodes,W,nNode,Nset);


%% Dispersion spectrum (k-omega form, only propagagtive wave branches are avialable)

sizeK = [5 0 10 15];
axesPos = funSpectrumK(Nset,kBA,kHDA,sizeK);


%% Dispersion spectrum (omega-k form, both propagagtive and evanescent wave branches are avialable)

sizeOmega = [5 0 30 15];
[axesPos1, axesPos2]= funSpectrumOmega(Nset,omegaBA,omegaHDA,sizeOmega,axesPos);


%% Characteristic coefficients (from computational homogenization)

% sizeR = [5 0 20 15];
% [refDeltabo, refDeltafl] = funCharCoef(HM,omegaHDA,sizeR);


%% Dynamic effective properties (from computational homogenization)

sizeRho = [5 0 10 15];
sizeC = [5 0 30 15];
Col = [0 0 0; 0.5 0.5 0.5];
funDynProp(HM,omegaHDA,sizeRho,sizeC,Col);

switch keyCell
    case 1
        save Data_dp_host
    case 2
        save Data_dp_NM
end

%% Plot modal displacement fields from Bloch analysis
Uk = kBA.U{end};          % k*(lambdaTh/2)/pi=1
ModeSel = [4 5 6];          % choose mode 1 to show
keyArrow = 1;           % show displacement direction
keyAll = 'Inclusion';   % only the displacement direction of the inclusion
funPlotModes(coords,elems,Uk,ModeSel,SizeM,keyArrow,keyAll,matrixNodes);



