%-------------------------------------------------------------------------%
% Background
% Computational homogenization under the lamina-level periodic boundary condition
% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

function [HM, kHDA, omegaHDA] = compHom(coords,presNodes,presIndices,Kqs,Mqs,Mff,Mpf,S,HM,W,Nset)

% get quasi-static effective elasticity, inertia and mass density
presCoords = zeros(2*length(presNodes),1);
for i = 1:2
   presCoords(i:2:end) = coords(presNodes,i);
end

% get in-plane and out-of-plane indices
hatIndices = [1 3 4 5];
tildeIndices = [2];
indices{1} = hatIndices;
indices{2} = tildeIndices;

Ar = W*1;                                          % reference surface area

th1 = clock;
[CM, DM, rhoM] = getCDRho(presCoords,indices,presIndices,Kqs,Mqs,Ar);  % CM and DM are completely in-plane
HM.CM = CM;
HM.DM = DM;
HM.rhoM = rhoM;

% get coupling coefficients
HM = getHJ(presCoords,indices,presIndices,Nset,Mff,Mpf,S,HM);          % Hc is completely in-plane
th2 = clock;

HM.dth = HM.dth + etime(th2,th1);                                          % add the time to compute effective properties
% dispersion analysis (complete k-omega form)
nMode = size(HM.resFreq,1) + 2;

th1 = clock;
[kHDA] = getDisperHomK(HM,Nset.nSample,nMode,Nset.dirWave,Nset.lambdaTh,Ar);
th2 = clock;

kHDA.dth = etime(th2,th1);                                                 % computation time

% dispersion analysis (simplified omega-k form)
omegaHDA = 1;
if Nset.method(2) == 1
    th1 = clock;
    [omegaHDA] = getDisperHomOmega(HM,Nset.nB,Nset.nOmega,Nset.freqLim,Nset.dirWave,Nset.lambdaTh,Ar);  
    th2 = clock;
    
    omegaHDA.dth = etime(th2,th1);                                         % computation time
end

end


%-------------------------------------------------------------------------%
% Function to compute quasi-static effective properties
% CM: mm, bb,1*1
% DM: mm,bb, 1*1
% rhoM: ll, rr, 2*2
%-------------------------------------------------------------------------%

function [CM, DM, rhoM] = getCDRho(presCoords,indices,presIndices,Kqs,Mqs,Ar)

% get in-plane projections of Kqs and Mqs
hatKqs = Kqs(indices{1},indices{1});
hatMqs = Mqs(indices{1},indices{1});

% compute in-plane quasi-static effective elasticity
% presCoords(2:2:end) is the same as Eta
CM.mm = (presCoords(1:2:end)'*hatKqs*presCoords(1:2:end))'/Ar;
CM.bb = ((presCoords(2:2:end).*presCoords(1:2:end))'*hatKqs*(presCoords(2:2:end).*presCoords(1:2:end)))'/Ar;

% compute in-plane quasi-static effective elasticity
DM.ll = (presCoords(1:2:end)'*hatMqs*presCoords(1:2:end))'/Ar;
DM.rr = ((presCoords(2:2:end).*presCoords(1:2:end))'*hatMqs*(presCoords(2:2:end).*presCoords(1:2:end)))'/Ar;

% compute quasi-static effective mass density
presIndice = length(presIndices);
Ip = ones(presIndice,1);

tildePresCoords = [presCoords(2:2:end);presCoords(2:2:end);];
tildePresCoords = tildePresCoords(presIndices);

for i = 1:2
    for j = 1:2
    rhoM.ll(i,j) = (Ip(indices{i})'*Mqs(indices{i},indices{j})*Ip(indices{j}))/Ar;    
    rhoM.rr(i,j) = ((tildePresCoords(indices{i}).*Ip(indices{i}))'*Mqs(indices{i},indices{j})*(tildePresCoords(indices{j}).*Ip(indices{j})))/Ar; 
        
    end
end
    
end


%-------------------------------------------------------------------------%
% Function to compute coupling coefficients
%-------------------------------------------------------------------------%
function HM = getHJ(presCoords,indices,presIndices,Nset,Mff,Mpf,S,HM)
Jc.lm = zeros(Nset.nMode,2);      % 1*2
Jc.rt = zeros(Nset.nMode,2);      % 1*2

Hc.lm = zeros(Nset.nMode,1);      % 1*1
Hc.rt = zeros(Nset.nMode,1);      % 1*1

presIndice = length(presIndices);
Ip = ones(presIndice,1);

tildePresCoords = [presCoords(2:2:end);presCoords(2:2:end);];
tildePresCoords = tildePresCoords(presIndices);

for iMode = 1:Nset.nMode
    
    % current mode
    fai = HM.Fai(:,iMode);
    
    % compute the coupling relation vector
    MC = S'*Mff*fai + eye(presIndice)*Mpf*fai;
    hatMC = MC(indices{1});           % in-plane projection
    
    % compute jc components
    lm = zeros(1,2);
    rt = zeros(1,2);
    for i = 1:2
         lm(i) = Ip(indices{i})'* MC(indices{i});
         rt(i) =  (tildePresCoords(indices{i}).*Ip(indices{i}))'* MC(indices{i});
    end
    
    Jc.lm(iMode,:) = lm;
    Jc.rt(iMode,:) = rt;
    
    % compute hc components
    Hc.lm(iMode,:) = hatMC'*presCoords(1:2:end);
	Hc.rt(iMode,:) = hatMC'*(presCoords(2:2:end).*presCoords(1:2:end));
    
end

% minimize HM
idActive = find(HM.resFreq<Nset.sf*Nset.freqLim); % only low-order localized modes are activated

HM.resFreq = HM.resFreq(idActive,:);
HM.Fai = HM.Fai(:,idActive);

HM.Jc.lm = Jc.lm(idActive,:);
HM.Jc.rt = Jc.rt(idActive,:);

HM.Hc.lm = Hc.lm(idActive,:);
HM.Hc.rt = Hc.rt(idActive,:);


end


%-------------------------------------------------------------------------%
% Function to compute dispersion curves and dynamic effective properties
% k-omega form
% CMmm, DMmm, CMbb, DMbb, rhoMll, Jclm, Hc 
% are to be used 
%-------------------------------------------------------------------------%
function [kHDA] = getDisperHomK(HM,nSample,nMode,dirWave,lambdaTh,Ar)

kmax = 2*pi / lambdaTh;
k = linspace(0,kmax,nSample);                       % wave number
eW = dirWave(1);                                    % in-plane wave direction
eC = 1;                                             % out-of-plane basis

% constant terms
resOmega2 = diag((HM.resFreq*2*pi).^2);

Kk = zeros(2+length(HM.resFreq));
Mk = diag(ones(2+length(HM.resFreq),1));
Freq = zeros(nMode,nSample);

for iK = 1:nSample   

    % fill in the matrix Kk
    Kk(1:2,1:2) = [k(iK)^2*eW*HM.CM.mm*eW        0;
                   0  k(iK)^4*eW*eW*HM.CM.bb*eW*eW*eC];
    
    Kk(3:end,3:end) = resOmega2;
    
    % fill in the matrix Mk, conj() is actually unnecessay here
    Mk(1:2,1:2) = [HM.rhoM.ll(1,1)+k(iK)^2*eW*HM.DM.ll*eW   0;
                   0             eC*HM.rhoM.ll(2,2)+k(iK)^4*eW*eW*HM.DM.rr*eW*eW*eC];    

    Mk(1,3:end) = transpose(HM.Jc.lm(:,1)-1i*k(iK)*eW*conj(HM.Hc.lm))/Ar;
    Mk(2,3:end) = transpose(eC*HM.Jc.lm(:,2)+k(iK)^2*eW*conj(HM.Hc.rt))/Ar;  
    
    Mk(3:end,1) = HM.Jc.lm(:,1)+1i*k(iK)*HM.Hc.lm*eW;
    Mk(3:end,2) = HM.Jc.lm(:,2)+k(iK)^2*HM.Hc.rt*eW*eW*eC;
    
    % solve the eigenvalue problem
    [~, Omega2] = eigs(Kk,Mk,nMode,'sr');
    
    Freq(:,iK) = sort(abs(sqrt(diag(Omega2)))/2/pi);
    
    % show process
    pcenK = (iK-1)/(nSample-1)*100;
    
    if rem(pcenK,1) == 0
    pcomp = fix(pcenK);
    fprintf('k-omega Homogenization-based Dispersion Analysis: %d%% completed \n ',pcomp)
    end
    
end

    kHDA.Normk = k / kmax;
    kHDA.Freq = Freq;

end


%-------------------------------------------------------------------------%
% Function to compute dispersion curves and dynamic effective properties
% omega-k form
% assume that body and flexural waves are separable, otherwise k-omega form
% is required
% CMmm, DMmm, CMbb, DMbb, rhoMll, Jclm, Hc are to be used
% Jclm and Hc are from different modes
%-------------------------------------------------------------------------%
function [omegaHDA] = getDisperHomOmega(HM,nB,nOmega,freqLim,dirWave,lambdaTh,Ar)

% generate the frequency axis
omegaHDA.Freq = linspace(0,freqLim,nOmega)';
% omegaHDA.Freq = ones(nOmega,1)*2005;

% store the dynamic effective properties
omegaHDA.dynP.rhoEll = zeros(nOmega,2);    % 1*2, 11, 22 components
omegaHDA.dynP.CEmm = zeros(nOmega,1);      % 1*1
omegaHDA.dynP.CEbb = zeros(nOmega,1);      % 1*1

% store dynamic cross-coupling coefficients
omegaHDA.R.lr = zeros(nOmega,1);           % 1*1 

% store corrected CEbb which takes Rlr into account
omegaHDA.dynP.CEbbC = zeros(nOmega,2);     % 1*2, + and -

% store Delta
omegaHDA.Delta.bo = zeros(nOmega,1);           % 1*1 
omegaHDA.Delta.fl = zeros(nOmega,1);           % 1*1 

% compute dynamic effective properties and dispersion braches for the body 
% and flexural waves
eW = dirWave(1);                           % in-plane wave direction
eC = 1;                                    % out-of-plane basis
omegaHDA.Normk = nan(nOmega,nB*2);         % 1-3: propagative, 4-6: evanescent

for iOmega = 1:nOmega
    omega = omegaHDA.Freq(iOmega)*2*pi;    % unit rad
    
    % dynamic effective mass density
    dhJJlm = HM.Jc.lm(:,1).*HM.Jc.lm(:,1) / Ar;% for the body wave
    omegaHDA.dynP.rhoEll(iOmega,1) = HM.rhoM.ll(1,1) + (omega^2./((HM.resFreq*2*pi).^2-omega^2))'*dhJJlm;
    
    dtJJlm = HM.Jc.lm(:,2).*HM.Jc.lm(:,2) / Ar; % for the flexural wave
    omegaHDA.dynP.rhoEll(iOmega,2) = HM.rhoM.ll(2,2) + (omega^2./((HM.resFreq*2*pi).^2-omega^2))'*dtJJlm; 
    
    % dynamic effective elasticity
    HHll = conj(HM.Hc.lm).*HM.Hc.lm / Ar;        
    omegaHDA.dynP.CEmm(iOmega) = HM.CM.mm - omega^2*(HM.DM.ll+(omega^2./((HM.resFreq*2*pi).^2-omega^2))'*HHll);
    
    HHrr = conj(HM.Hc.rt).*HM.Hc.rt / Ar;        
    omegaHDA.dynP.CEbb(iOmega) = HM.CM.bb - omega^2*(HM.DM.rr+(omega^2./((HM.resFreq*2*pi).^2-omega^2))'*HHrr);

    % dynamic cross-coupling coefficients
    jHlr = HM.Jc.lm(:,2).*conj(HM.Hc.rt) / Ar;        
    omegaHDA.R.lr(iOmega) =  (omega^2./((HM.resFreq*2*pi).^2-omega^2))'*jHlr;
    
    % corrected CEbb
    rholl = eC*eC * omegaHDA.dynP.rhoEll(iOmega,2);
    Cbb = eW*eW * omegaHDA.dynP.CEbb(iOmega) * eW*eW;
    Rlr = eC* omegaHDA.R.lr(iOmega) * eW*eW;
    
    omegaHDA.dynP.CEbbC(iOmega,1) = (omega*Rlr + sqrt((omega*Rlr)^2 + rholl*Cbb))^2 / (rholl*Cbb);
    omegaHDA.dynP.CEbbC(iOmega,2) = (omega*Rlr - sqrt((omega*Rlr)^2 + rholl*Cbb))^2 / (rholl*Cbb);
    
    omegaHDA.dynP.CEbbC(iOmega,:) = omegaHDA.dynP.CEbbC(iOmega,:)*omegaHDA.dynP.CEbb(iOmega);
    
    % form characteristic dynamic stiffness matrices
    % 0th-order
    rhoEWlldh = omega^2*omegaHDA.dynP.rhoEll(iOmega,1);  
    rhoEWlldt = omega^2*omegaHDA.dynP.rhoEll(iOmega,2);
    
    K0 = -[rhoEWlldh 0; 0 rhoEWlldt];
     
    % 1st-order
    K1 = [0 0; 0 0];
    
    % 2nd-order
    CEWmm = eW * omegaHDA.dynP.CEmm(iOmega) * eW;
    REWlr = -eW * 2*omega^2*omegaHDA.R.lr(iOmega) * eW*eW * eC;
    
    K2 = [CEWmm 0; 0 REWlr];
    
    % 3rd-order
    K3 = [0 0; 0 0];
    
    % 4th-order
    CEWbb = eW*eW * omegaHDA.dynP.CEbb(iOmega) * eW*eW * eC;
    
    K4 = [0 0; 0 CEWbb];
   
    % Delta to judge existence of the real solution
    % body wave
    % split wave equation k^2*CEWmm - rhoEWlldh = 0
    omegaHDA.Delta.bo(iOmega) = 4*CEWmm*rhoEWlldh;
    
    % flexural wave
    % split wave equation k^4*CEWbb + k^2*REWlr - rhoEWlldt = 0
    omegaHDA.Delta.fl(iOmega) = REWlr^2 + 4*CEWbb*rhoEWlldt;

    % solve the eigenvalue problem
    k = polyeig(K0,K1,K2,K3,K4);
    normk = k*lambdaTh /2/pi;

    % select appropriate k, default real(k)>0 and imag(k)>0 due to the symmetry   
    % filter normk with modulus over 1
    [~,index] = find(abs(normk)'<=1);
    normkSet = roundn(normk(index),-4);           % accuracy, 4 decimal digits
    normkSet = unique(sort(abs(real(normkSet)) + abs(imag(normkSet))*1i));
    nNormk = length(normkSet);
    
    % propagative 
    [~,indexProp] = find(abs(imag(normkSet))'==0);
    normkProp = normkSet(indexProp);
    
    for iP = 1:length(normkProp)   
        omegaHDA.Normk(iOmega,iP) = normkProp(iP);
    end
    
    % evanescent
    indexEva = setdiff(1:nNormk,indexProp);
    normkEva = normkSet(indexEva);
    
    for iE = 1:length(normkEva) 
        omegaHDA.Normk(iOmega,nB+iE) = normkEva(iE);
    end
    
    % show process
    pcenOmega = (iOmega-1)/(nOmega-1)*100;
    
    if rem(pcenOmega,1) == 0
    pcomp = fix(pcenOmega);
    fprintf('omega-k Homogenization-based Dispersion Analysis: %d%% completed \n ',pcomp)
    end
    
end


end





