%-------------------------------------------------------------------------%
% Background
% Dispersion analyis under the Bloch-Floquet boundary condition
% Created by Lei, TU/e, 2018
%-------------------------------------------------------------------------%

function [kBA, omegaBA]= blochAnalysis(K,M,boundNodes,W,nNode,Nset)

% allocate the nodes
souNodes = [boundNodes{1}]';                    % source nodes
depNodes = [boundNodes{2}]';                       % dependent nodes
exNodes = setdiff([1:nNode]',[souNodes; depNodes]); % extra nodes, part of the dependent node set

% dispersion analysis (k-omega form)
tb1 = clock;
kBA = getDisperBlochK(Nset,souNodes,depNodes,exNodes,K,M,W);
tb2 = clock;
kBA.dtb = etime(tb2,tb1); % time for Bloch analysis

% dispersion analysis (omega-k form)
omegaBA = 1;
if Nset.method(1) == 1
    tb1 = clock;
	omegaBA = getDisperBlochOmega(Nset,souNodes,depNodes,exNodes,K,M,W);
    tb2 = clock;
    omegaBA.dtb = etime(tb2,tb1); % time for Bloch analysis
end
end

%-------------------------------------------------------------------------%
% Function to compute dispersion curves
% k-omega form 
%-------------------------------------------------------------------------%

function kBA = getDisperBlochK(Nset,souNodes,depNodes,exNodes,K,M,W)

% generate the wave number axis
nMode = Nset.nMode;                       % total number of selected modes
nSample = Nset.nSample;                              % sampling number

kmax = 2*pi / Nset.lambdaTh;
k = linspace(0,kmax,nSample);                        % wave number
kX = Nset.dirWave(1) * k;                          % only x direction is useful, relative position is identical to [W, 0]

% get the frequency
U = cell(1,nSample);
Freq = zeros(nMode,nSample);

% get numer of node pairs
nNodesP = length(souNodes);                            % number of X-boundary nodes, left-right bounds

% get DOF sets
indNodes = [souNodes; exNodes];                      % independent nodes
 
indDofs = getDofs(indNodes);
indDof = length(indDofs);

depDofs = getDofs(depNodes);
depDof = length(depDofs);

totDof = indDof+depDof;

for iK = 1:nSample    
    % form the dependency matrix Cdi
    kx = kX(iK);
    Tdi = funTdi(indDofs,depDofs,nNodesP,kx,W);
    
    % partition K and M as [ii,id;di,dd]
    Kii = K(indDofs,indDofs);
    Kid = K(indDofs,depDofs);
    Kdi = K(depDofs,indDofs);
    Kdd = K(depDofs,depDofs);

    Mii = M(indDofs,indDofs);
    Mid = M(indDofs,depDofs);
    Mdi = M(depDofs,indDofs);
    Mdd = M(depDofs,depDofs);
    
    % remove dependent nodes (KEii*ui + MEii*ai = fiEii)
    KEii = Kii +Kid*Tdi + Tdi'*Kdi +Tdi'*(Kdd*Tdi);    % .' is equivalent to transpose(), ' is equivalent to transpose(conj())
    MEii = Mii +Mid*Tdi + Tdi'*Mdi +Tdi'*(Mdd*Tdi);

    % solve the eigenvalue problem
    [indU, Omega2] = eigs(KEii,MEii,nMode,'sm');      % results at free DOFs
    
    depU = Tdi * indU;
    
    % collect frequency
    [Freq(:,iK), idFreq]= sort(abs(sqrt(diag(Omega2)))/2/pi);
    
    % collect results at all nodes
    totU = zeros(totDof,nMode);
    totU(indDofs,:) = indU(:,idFreq);
    totU(depDofs,:) = depU(:,idFreq);
    
    U{iK} = totU; 
    
    % show process
    pcenK = (iK-1)/(nSample-1)*100;
    
    if rem(pcenK,1) == 0
    pcomp = fix(pcenK);
    fprintf('k-omega Bloch Analysis: %d%% completed \n ',pcomp)
    end
    
end

kBA.Normk = k / kmax;
kBA.Freq = Freq;
kBA.U = U;

end


function Tdi = funTdi(indDofs,depDofs,nNodesP,kx,W)

% independent: left and others, dependent: right
indDof = length(indDofs);
depDof = length(depDofs);

% initialize
Tdx = exp(1i*kx*W)*[ 1 0; 0 1];  
Tdi = zeros(depDof,indDof);

% for right nodes
for iright = 1:nNodesP
    Tdi(2*iright-1:2*iright,2*iright-1:2*iright) = Tdx;

end

end

%-------------------------------------------------------------------------%
% Function to compute dispersion curves
% omega-k form 
%-------------------------------------------------------------------------%
function omegaBA = getDisperBlochOmega(Nset,souNodes,depNodes,exNodes,K,M,W)

% get DOF sets
souDofs = getDofs(souNodes);      
depDofs = getDofs(depNodes);
 
exDofs = getDofs(exNodes);
  
% generate the frequency axis
omegaBA.Freq = linspace(0,Nset.freqLim,Nset.nOmega)';
% omegaBA.Freq = ones(Nset.nOmega,1) * 2504;

omegaBA.Normk = nan(Nset.nOmega,Nset.nB*2);     % 1-3: propagative, 4-6: evanescent

for iOmega = 1:Nset.nOmega
    omega = omegaBA.Freq(iOmega)*2*pi;    % unit rad
    
    % compute the dynamic stiffness matrix
    KM = K - omega^2*M;
    
    % partition the matrix
    KMss = KM(souDofs,souDofs);
    KMdd = KM(depDofs,depDofs);
    KMee = KM(exDofs,exDofs);
    
    KMsd = KM(souDofs,depDofs);
    KMds = KM(depDofs,souDofs);
    
    KMse = KM(souDofs,exDofs);
    KMes = KM(exDofs,souDofs);
    
    KMde = KM(depDofs,exDofs);
    KMed = KM(exDofs,depDofs);
    
    % compute 0th-order, 1th-order, 2th-order terms
    KM0 = KMds - KMde*inv(KMee)*KMes;
    KM1 = KMss - KMse*inv(KMee)*KMes + KMdd - KMde*inv(KMee)*KMed;
    KM2 = KMsd - KMse*inv(KMee)*KMed;
    
    % solve the eigenvalue problem
    [~,Lambda] = polyeig(KM0,KM1,KM2);
    k = log(Lambda) / (1i*W);
    normk = k*Nset.lambdaTh /2/pi;
    
    % select appropriate k, default real(k)>0 and imag(k)>0 due to the symmetry   
    % filter normk with modulus over 1
    [~,index] = find(abs(normk)'<=1);
    normkSet = roundn(normk(index),-4);           % accuracy, 4 decimal digits
    normkSet = unique(sort(abs(real(normkSet)) + abs(imag(normkSet))*1i));
    nNormk = length(normkSet);
    
    % propagative 
    [~,indexProp] = find(abs(imag(normkSet))'==0);
    normkProp = normkSet(indexProp);
    
    for iP = 1:length(normkProp)   
        omegaBA.Normk(iOmega,iP) = normkProp(iP);
    end
    
    % evanescent
    indexEva = setdiff(1:nNormk,indexProp);
    normkEva = normkSet(indexEva);
    
    for iE = 1:length(normkEva) 
        omegaBA.Normk(iOmega,Nset.nB+iE) = normkEva(iE);
    end
    
    % show process
    pcenOmega = (iOmega-1)/(Nset.nOmega-1)*100;
    
    if rem(pcenOmega,1) == 0
    pcomp = fix(pcenOmega);
    fprintf('omega-k Bloch Analysis: %d%% completed \n ',pcomp)
    end
    
end

end



