%------------------------------------------------------------------------%
%  Function: Smooth5                                                      %
%------------------------------------------------------------------------%
%  Purpose : Smooth an array with multiple sequences over n positions    %
%------------------------------------------------------------------------%
%  External functions used:                                              %
%------------------------------------------------------------------------%
%  Creation / modification history:                                      %
%                                                                        %
%  2006-01-10 - v1.00 - B.A.Witvliet - creation                          %
%  2014-03-25 - v1.01 - B.A.Witvliet - addition to accept log data       %
%  2021-08-18 - v1.02 - B.A.Witvliet - more flexibe windowing            %
%  2022-01-03 - v1.03 - B.A.Witvliet - median as smoothing filter        %
%------------------------------------------------------------------------%

function UrxOut = Smooth5(UrxIn,FilterType,FilterSize,DataType)
%example:    Power_dBm_smoothed = Smooth5(Power_dBm,'Gaussian',11,'Log')

% - Make sure that the input vector is oriented correctly -
if size(UrxIn,2) > size(UrxIn,1)
    UrxIn = UrxIn';
end

% - Define smoothing filter -
median_flag = 0;
switch FilterType
    case 'Gaussian',       filtervector = gausswin(      FilterSize);
    case 'BlackmanHarris', filtervector = blackmanharris(FilterSize);
    case 'Rectangle',      filtervector = ones(FilterSize,1);
    case 'median',         filtervector = ones(FilterSize,1);
                           median_flag  = 1;
    otherwise
        err = 1; 
        disp('This smoothing filter is unknown'); 
        filtervector = [1];
end

% - Check if windowsize greater than one -
windowsize = length(filtervector);
if windowsize == 1
    UrxOut = UrxIn; %no smoothing
else

    %--------------------------
    %  Smoothing starts here  -
    %--------------------------
    
    % - Median smoothing (no windowing)
    if median_flag
       
        UrxOut = medfilt1(UrxIn,FilterSize);
 
    else

        % - Add heading and trailing dummy cells -
        ldata   = length(UrxIn);
        hwindow = ceil(windowsize/2); %half window length
        for i = 1: hwindow
            UrxIn = [UrxIn(1,:); UrxIn; UrxIn(ldata,:)];
        end
        
        % - Smoothing by windowing and calculating linear mean for lineair of logaritmic values -
        switch DataType
            case 'Lin'
                UrxIn = filter(filtervector,sum(filtervector),UrxIn);
            case 'Log'
                UrxIn_lin  = 10.^(UrxIn./10);
                UrxIn_lin2 = filter(filtervector,sum(filtervector),UrxIn_lin); %filtervector, weighting, inputdata
                UrxIn      = 10*log10(UrxIn_lin2);
            otherwise
                disp('Smoothing: Unknown datatype specified, only <Log> and <Lin> accepted'); 
                return;
        end %switch

        % Removing heading and trailing dummy cells + delay caused by filter -
        ldata  = length(UrxIn);
        re     = abs(windowsize-2*hwindow); %to correct for rounding error
        UrxOut = UrxIn(windowsize+re:ldata-1,:);
    
    end %if median_flag


end %if windowsize


% - Cleanup memory -
clear filtervector windowsize ldata hwindow i re;