% ================================================================ %
%|  Program name:     CALCULATE COMPLEX FREQUENCY SPECTRUM        |%
%|----------------------------------------------------------------|%
%|  Purpose:          Calculate power an phase spectrogram        |%
%|                    of time series measurement data from the    |%
%|                    form the HPSDR receiver.                    |%
%|----------------------------------------------------------------|%
%|  Author:           Ben Witvliet PE5B                           |%
%|  Created:          2015-01-30                                  |%
%|  Last modified:    2016-02-14                                  |%
% ================================================================ %

function [Header,Data]= Calc_Freq_Spectrum(Settings,Header,Data)

% - Truncate data to get complete blocks -
nrofblocks = floor(length(Data.IQrx1)/Settings.FFT_blocksize);
Data.IQrx1 = Data.IQrx1(1:nrofblocks*Settings.FFT_blocksize);
Data.IQrx2 = Data.IQrx2(1:nrofblocks*Settings.FFT_blocksize);
Data.Time  = Data.Time( 1:nrofblocks*Settings.FFT_blocksize);

% - Reshape Data.Time series data to blocks of length <Settings.FFT_blocksize> -
IQ1     = reshape(Data.IQrx1,Settings.FFT_blocksize,[]);
IQ2     = reshape(Data.IQrx2,Settings.FFT_blocksize,[]);
TimeFFT = reshape(Data.Time, Settings.FFT_blocksize,[]);

% - Choose time in middle of block -
mid     = round(Settings.FFT_blocksize/2);
TimeFFT = TimeFFT(mid,:);

% - Apply window function -
switch Settings.FFT_window
    case 'Blackman-Harris-999-bugfix'
        w = Calc_Blackman_Harris_Window;
        W = w'*ones(1,nrofblocks);
        IQ1w = IQ1.*W;
        IQ2w = IQ2.*W;
    case 'Blackman-Harris'
        W = blackmanharris(Settings.FFT_blocksize)*ones(1,nrofblocks);
        IQ1w = IQ1.*W;
        IQ2w = IQ2.*W;
    case 'Bohman'
        W = bohmanwin(Settings.FFT_blocksize)*ones(1,nrofblocks);
        IQ1w = IQ1.*W;
        IQ2w = IQ2.*W;
    case 'Chebyshev' %-100dB sidelobe version
        W = chebwin(Settings.FFT_blocksize)*ones(1,nrofblocks);
        IQ1w = IQ1.*W;
        IQ2w = IQ2.*W;
    case 'Flattop' 
        W = chebwin(Settings.FFT_blocksize)*ones(1,nrofblocks);
        IQ1w = IQ1.*W;
        IQ2w = IQ2.*W;
    case 'Nuttal'
        W = chebwin(Settings.FFT_blocksize)*ones(1,nrofblocks);
        IQ1w = IQ1.*W;
        IQ2w = IQ2.*W;
    otherwise %'none'
        IQ1w = IQ1;
        IQ2w = IQ2; 
        disp('Warning [Calc_Freq_Spectrum]: specified window type unknown - no window applied in FFT');
end

% - Calculate Complex Amplitude Frequency Spectrum -
A1 = fft(IQ1w)/Settings.FFT_blocksize/sqrt(2);  %[Urms]
A2 = fft(IQ2w)/Settings.FFT_blocksize/sqrt(2);  %[Urms]  

% -  Calculate Relative-Power Spectrum -
Pwr1 = 20*log10(abs(A1))-10*log10(50);  %[dBW]
Pwr2 = 20*log10(abs(A2))-10*log10(50);  %[dBW]

% -  Calculate Phase Spectrum -
Ph1deg = angle(A1)*180/pi; %[deg]
Ph2deg = angle(A2)*180/pi;

% - Create frequency vector -
fstep = Header.samplerate/(Settings.FFT_blocksize-1);
Freqs = Header.centerfreq+([0:fstep:Header.samplerate]-Header.samplerate/2)/1000; %[kHz]

% - ? SWITCH UPPER AND LOWER HALF OF SPECTRUM - DON'T KNOW WHY THIS IS NEEDED -
half   = round(Settings.FFT_blocksize/2);
Pwr1   = [Pwr1(  half+1:Settings.FFT_blocksize,:); Pwr1(  1:half,:)]; 
Pwr2   = [Pwr2(  half+1:Settings.FFT_blocksize,:); Pwr2(  1:half,:)]; 
Ph1deg = [Ph1deg(half+1:Settings.FFT_blocksize,:); Ph1deg(1:half,:)]; 
Ph2deg = [Ph2deg(half+1:Settings.FFT_blocksize,:); Ph2deg(1:half,:)];

% - Calculate phase difference -
DeltaPhi = Ph1deg-Ph2deg;

% - Keep phase within [-180; 180] degrees -
Pos           = find(DeltaPhi>+180);  
DeltaPhi(Pos) = DeltaPhi(Pos)-360;
Pos           = find(DeltaPhi<-180);  
DeltaPhi(Pos) = DeltaPhi(Pos)+360;

% - Save results in structured array -
Data.TimeFFT  = TimeFFT;
Data.Freqs    = Freqs;
Data.Pwr1     = Pwr1;
Data.Pwr2     = Pwr2;
Data.DeltaPhi = DeltaPhi;

