%==============================================================%
%                   SCN AND IN FILTERING                       %
% ============================================================ %
%   author: B.A.Witvliet, University of Twente                 % 
%   date:   10 january 2023                                    %
%   code:   Matlab 2021b Update 1 (Nov. 2021)                  %
% ============================================================ %
%   When resusing this code, please acknowledge the original   %
%   author.                                                    %
%   When reusing this dataset, please cite the original        %
%   publication, which contains the associated background      %
%   information:                                               %
%   Witvliet BA, Alsina-Pagès RM, Van Maanen E, Laanstra GJ.   %
%   Design and validation of probes and sensors for the        %
%   characterization of magneto-ionic radio wave propagation   %
%   on Near Vertical Incidence Skywave paths. Sensors.         %
%   2019 19(11).                                               %
%==============================================================%

% - Clear memory -
close all; clear; clc;


% - USER FIGURE SELECTION -
fignr = 6;  
% 1 = waterfall raw data
% 2 = adaptive filter explanation
% 3 = waterfall after adaptive filter (SCN filter)
% 4 = time graph with and without impulsive noise filter (IN filter)
% 5 = waterfall after SCN and IN filter
% 6 = histogram noise before and after all filtering



% - Data directory -
InputDir = '.\DATA\';  

% All measurements:
% 11-2-2016  07:45:13 - 18:02:24  #1-5
% 12-2-2016  07:40:26 - 18:03:37  #6-9

% - User defined constants -
BlankValue = NaN;    %for calculations
BlankValue = -999;   %for visualizations
dyn_range = 60;                     %[dB]  waterfall dynamic range
minscale  = -165;                   %[dBr] waterfall min value
maxscale  =  minscale + dyn_range;  %[dBr] waterfall max value
fontsz    = 32;                     %[pnt] font size
lwd       = 2.5;                    %[pnt] line width
dkyellow  = [1 0.5 0];              %[RGB] color
dkgreen   = [0 0.6 0];              %[RGB] color

% - Presets -
Settings.FFT_window    = 'Blackman-Harris';  %window used in graphs %%% TEMP BUGFIX %%%%%
Settings.FFT_blocksize = 1000;               %nr of FFT bins
noise_hour             = 22;                 %measurement hour
tzoom                  = 1;                  %time zoom on
noisemargin_SCN        = 2.5;                %[dB] margin above the asymptote for SCN filter

% - Processing -
switch fignr
    case 1   %-- waterfall unfiltered --
        filt_SCN          = 0;   %adaptive SCN filter off
        filt_IN           = 0;   % impulsive noise filter off
        show_thresh_graph = 0;   % show adaptive filter threshold determination for one frequency scan (example)
        show_waterfall    = 1;   % show waterfall 
        show_hist         = 0;   % show histogram 

    case 2   %-- threshold graph --
        filt_SCN          = 1;   % adaptive SCN filter on
        filt_IN           = 0;   % impulsive noise filter off
        show_thresh_graph = 1;   % show adaptive filter threshold determination for one frequency scan (example)
        show_waterfall    = 0;   % show waterfall 
        show_hist         = 0;   % show histogram 
    
    case 3  %-- waterfall SCN filtered --
        filt_SCN          = 1;   % adaptive SCN filter on
        filt_IN           = 0;   % impulsive noise filter off
        show_thresh_graph = 0;   % show adaptive filter threshold determination for one frequency scan (example)
        show_waterfall    = 1;   % show waterfall 
        show_hist         = 0;   % show histogram 
    
    case 4  %-- timegraph SCN + IN filtered -- (time!)
        filt_SCN          = 1;   % adaptive SCN filter on
        filt_IN           = 1;   % impulsive noise filter on
        show_thresh_graph = 0;   % show adaptive filter threshold determination for one frequency scan (example)
        show_waterfall    = 0;   % show waterfall 
        show_hist         = 0;   % show histogram 

    case 5  %-- waterfall SCN + IN filtered -- (waterfall!)
        filt_SCN          = 1;   % adaptive SCN filter on
        filt_IN           = 1;   % impulsive noise filter on
        show_thresh_graph = 0;   % show adaptive filter threshold determination for one frequency scan (example)
        show_waterfall    = 1;   % show waterfall 
        show_hist         = 0;   % show histogram 
    
    case 6  % -- histogram --
        filt_SCN          = 1;   % adaptive SCN filter on
        filt_IN           = 0;   % impulsive noise filter on  
        show_thresh_graph = 0;   % show adaptive filter threshold determination for one frequency scan (example)
        show_waterfall    = 0;   % show waterfall  
        show_hist         = 1;   % show histogram 

    otherwise
        disp('unknown figure number');
        return;
end



% ============================
%  READ SELECTED DATA INTERVAL
% ============================

% - Get filelist -
load([InputDir '/FileList_Blanked.mat']);  

switch noise_hour
	case 22
        filenr = 9;  
        starttime = datenum('12-Feb-2016 16:59:59');
        duration  = 60*60+2;           %[s]
        Settings.timeInterval = 10*60; %[s] time markers     
end

% - Read data from file -
FileName = FileList.Name{filenr};
FullName = [InputDir '/' FileName '.mat'];
disp(['Loading data from ' FileName]);
load(FullName);
%%% THIS IS PROPERLY CALIBRATED IQ-DATA, AMPLITUDE IS CALIBRATED WITH
%%% EXTERNAL SIGNAL SOURCE

if tzoom
    % - Select time interval -
    Pos = find(Data.Time >= starttime);
    Pos1 = Pos(1);                                 %first sample number of time interval in file
    Pos2 = Pos1 + duration * Header.samplerate;    %last sample number of time interval in file
    clear Pos;
    Data.Time =  Data.Time(Pos1:Pos2);
    Data.IQrx1 = Data.IQrx1(Pos1:Pos2); 
    Data.IQrx2 = Data.IQrx2(Pos1:Pos2);
    disp(['Data selected from ' datestr(min(Data.Time)) ' to ' datestr(max(Data.Time)) '.']); disp(' ');
end


% ===============================
% TIME-FREQUENCY ANALYSIS (FFT's)
% ===============================

% - User information -
disp('Calculating FFTs...');

% - Calculate frequency spectrum -
[Header,Data]= Calc_Freq_Spectrum(Settings,Header,Data);
% Result is in dBW

% - Add PSD of both channels -
Pwr     = 10*log10(10.^(Data.Pwr1/10) + 10.^(Data.Pwr2/10));  %[dBW]
Freqs   = Data.Freqs;                                         %[kHz]
Time    = Data.Time;
TimeFFT = Data.TimeFFT;
clear Data; %memory clean-up

% - Calculate PSD spectral density -
nr_of_bins = length(Freqs);
RBW        = 3000;                %[Hz] receiver bandwidth == IQ sample rate
BW         = RBW / nr_of_bins;    %[Hz] bandwidth of one bin
PSD        = Pwr - 10*log10(BW);  %[dBW/Hz]
clear Pwr;

% - Blank the filter edges of the passband -
edge          = 50;
passband      = size(PSD,1);
PSD(1:edge,:) = BlankValue;
PSD(passband-edge+1:passband,:) = BlankValue;

%- Make vector containing the blocknumbers -
nrofblocks = size(PSD,2);
Blocks     = 1: nrofblocks;

%- Create meshgrid for plotting the waterfall
[Blocks2D,Freqs2D] = meshgrid(Blocks,Freqs);



%-------------------------------------%
%  REMOVE Single-Carrier Noise (SCN)  %
%-------------------------------------%
if filt_SCN

    % - User information -
    disp('Apply adaptive SCN filter...');

    % B.A.Witvliet, 9-9-2018
    % nroffreqs:  	number of frequency bins in each FFT
    % PSD:          array of raw measurement samples after FFT [PSD in dBW/Hz]
    %               e.g. 999 freqs x 7000 measurements 
   
    % - For each FFT, sort on the amplitudes it contains -
    P_sorted = sort(PSD,1);      
    
    % - Find the reference number of the FFT bins corresponding to 40%, 50% and 60% values -
    nroffreqs = size(PSD,1);
    m40 = round(nroffreqs*0.4);  % 40 percent position in sorted sequence
    m50 = round(nroffreqs*0.5);  % 50 percent 
    m60 = round(nroffreqs*0.6);  % 60 percent 
    
    % - Find the median PSD of each FFT -
    PSD_median = P_sorted(m50,:);        % median value of each FFT
  
    % - For each FFT, find the slope of the asymptote at the median value -
    Slope = (P_sorted(m60,:)-P_sorted(m40,:))/(m60-m40);
    
    % - For each FFT, create a tangent of values -
    m      = 1: nroffreqs;
    P_line = PSD_median + (m-m50)' * Slope;
    
    % - Find the first position in the FFT in which the PSD exceeds this tangent -
    %  (that is: which a small noise margin added)
    Pos            = find (P_sorted > P_line + noisemargin_SCN); %find positions where the measurement value exceeds the line
    P_sorted2      = P_sorted;
    P_sorted2(Pos) = -999;                  %null these time-frequency bins
    Threshold      = max(P_sorted2,[],1);   %find the highest value that is just below the line        

    % - Create a 2D threshold plane, one threshold value per FFT -
    Threshold_2D = ones(nroffreqs,1)*Threshold;
    
    % - Delete the detected signal cells from the spectrogram -
    SigsPos                  = find(PSD > Threshold_2D);
    PSD_SCN_filtered         = PSD;
    PSD_SCN_filtered(SigsPos)= BlankValue;

end %filt_SCN


%======================================
%  SHOW ADAPTIVE FILTER THRESHOLD GRAPH
%======================================
if show_thresh_graph && filt_SCN

    % - User information -
    disp('Show adaptive filter threshold determination...');

    % - Prepare graph space -
    scrsz  = get(0,'ScreenSize');  
    graph  = figure('Position',scrsz,'Color','white');    
    Axe{1} = [.12  .19  .79  .78];
    axes('position',Axe{1},'FontSize',fontsz); 
    set(gca,'FontSize',fontsz,'FontWeight','normal');
    hold on; grid on; box on;

    % - Select one scan and use this data -
    scannr = 73;
    PSD_n = P_sorted(:,scannr);
    
    % - Plot graph -
    plot(m,PSD_n,'Color','blue','Linewidth',lwd);   %sorted PSD
    
    % - Tangent line -
    Slope_line = PSD_n(m50) + (m-m50)' * Slope(scannr);
    plot(m,Slope_line',                  'Color','red','Linewidth',lwd,'Linestyle','--');  %tangent
    plot(m,Slope_line' + noisemargin_SCN,'Color','red','Linewidth',lwd,'Linestyle','--');
    
    % - 40% - 50% - 60% markers -
    plot([m40 m50 m60],[PSD_n(m40) PSD_n(m50) PSD_n(m60)],'Color','black', ...
        'LineWidth',lwd+1,'Marker','+','MarkerSize',20,'Linestyle','none');   %40% 50% 60% markers
    text(m40,PSD_n(m40)-2,' 40%','FontSize',fontsz,'HorizontalAlignment','center');
    text(m50,PSD_n(m50)-2,' 50%','FontSize',fontsz,'HorizontalAlignment','center');
    text(m60,PSD_n(m60)-2,' 60%','FontSize',fontsz,'HorizontalAlignment','center');
    
    % - Show threshold -
    plot(713*[1 1],[-200 0],'Color','black','Linewidth',lwd,'Linestyle','--'); %<<<temp for paper<<<<
    text(710,-176.8,'threshold','FontSize',fontsz,'Rotation',90,'BackgroundColor','white');
    
    % - Graph make-up -
    min_y = round((PSD_n(m50)-20)/10)*10;
    axis([1 length(PSD_n) min_y min_y+50]);
    xlabel('Frequency bins (sorted)','FontSize',fontsz);
    ylabel('PSD \rightarrow','FontSize',fontsz);
    yticks(-200:10:0);
   
end %thresh_graph


%-------------------------------%
%  REMOVE Impulsive Noise (IN)  %
%-------------------------------%
if filt_IN

    % - User information -
    disp('Show impulsive noise timegraph...');

    % - Prepare graph space -
    scrsz  = get(0,'ScreenSize');  
    graph  = figure('Position',scrsz,'Color','white');    
    Axe{1} = [.12  .19  .69  .78];
    axes('position',Axe{1},'FontSize',fontsz); 
    set(gca,'FontSize',fontsz,'FontWeight','normal');
    hold on; grid on; box on;

    % - Plot median FFT PSD values (timeseries before IN filter) -
    plot(TimeFFT,PSD_median,'red');  %median of the FFT
    hold on;

    % - User information -
    disp('Apply impulsive noise blanker...');

    % - For each FFT, calculated the median PSD -
    PSD_median = median(PSD_SCN_filtered,1,'omitnan');     %[dBW/Hz]
    
    % - Make a smoothly varying median by applying a window -
    PSD_median_sm = Smooth6(PSD_median,'median',72,'Lin');    %[dBW/Hz]
    PSD_thresh_IN = PSD_median_sm + 1;                        %[dBW/Hz]
    
    % - Plot IN filter PSD threshold -
    plot(TimeFFT,PSD_thresh_IN,'Color','black','LineStyle','-','LineWidth',1.5);
    axis([min(TimeFFT+20/3600/24) max(TimeFFT) -170 -130]);

    % - Filter IN in time series -
    PSD_IN_filtered_timeseries = PSD_median;
    Pos_IN = find(PSD_IN_filtered_timeseries > PSD_thresh_IN');
    PSD_IN_filtered_timeseries(Pos_IN) = NaN;

    % - User information -
    disp('Show time series without impulsive noise...');

    % - Plot IN filtered PSD in time series -
    plot(TimeFFT,PSD_IN_filtered_timeseries,'blue');
        
    %- Time axis make-up -
    TimeTicks  = floor(TimeFFT) : Settings.timeInterval/3600/24 : ceil(TimeFFT);  %marker every 10 min.
    set(gca,'XTick',TimeTicks,'TickDir','out','LineWidth',1.5);
    datetick('x','HH:MM','keeplimits') ;
    xlabel('time \rightarrow');
    
    %- Amplitude axis make-up -
    set(gca,'YTick',-200: 10: 100,'TickDir','out','LineWidth',1.5);
    ylabel('PSD  [dBW/Hz] \rightarrow');

    %- Legend -
    legend({' impulsive noise', ' threshold', ' background'},'Fontsize',fontsz-4);

    % - Remove impulsive noise FFT's -
    if filt_IN
        PSD_SCN_IN_filtered = PSD_SCN_filtered;
        for i =1 : length(Pos_IN)
            PSD_SCN_IN_filtered(:,Pos_IN(i)) = BlankValue*ones(1000,1);
        end
    end

end  %if Filt_IN



% ==============
% SHOW WATERFALL
% ==============
if show_waterfall

    % - User information -
    disp('Show waterfall...');
    
    % - Prepare graph space -
    scrsz  = get(0,'ScreenSize');  
    graph  = figure('Position',scrsz,'Color','white');    
    Axe{1} = [.12  .19  .79  .78];
    axes('position',Axe{1},'FontSize',fontsz); 
    set(gca,'FontSize',fontsz,'FontWeight','normal');
    hold on; grid on; box on;

    % - Select the right data set -
    if filt_SCN
        if filt_IN
            PSD_Sel = PSD_SCN_IN_filtered;  % both SCN and IN filtered
        else
            PSD_Sel = PSD_SCN_filtered;     % only SCN filtered
        end
    else
        PSD_Sel = PSD;                  % raw data
    end
    
    %- Plot the PSD waterfall -
    h = surf(Blocks2D,Freqs2D,PSD_Sel);
    set(h,'EdgeColor','none','FaceColor','interp');
    
    %- Add colorbar -
    c = colorbar('FontSize',fontsz,'FontWeight','normal');
    c.Label.String = 'Power spectral density  [dBW/Hz] \rightarrow';
    
    %- Frequency axis make-up -
    f_edge = edge/passband*(max(Freqs)-min(Freqs));
    axis([1 nrofblocks min(Freqs)+f_edge max(Freqs)-f_edge]);
    ylabel('Frequency  [kHz] \rightarrow');
    xlabel('Local time [UTC+1] \rightarrow');
    
    %- Time axis make-up -
    TimeTicks  = floor(TimeFFT) : Settings.timeInterval/3600/24 : ceil(TimeFFT);
    BlockTicks = 1+round( (TimeTicks-min(TimeFFT))*24*3600*Header.samplerate/Settings.FFT_blocksize);
    BlockTicks = BlockTicks(find(BlockTicks>=1));
    BlockTicks = BlockTicks(find(BlockTicks<=nrofblocks));
    set(gca,'XTick',BlockTicks,'TickDir','out','LineWidth',1.5);
    set(gca,'XTickLabel',datestr(TimeFFT(BlockTicks),'HH:MM') );
    
    %- Z-axis make-up -
    caxis([minscale maxscale]);
    colormap('Jet');

end %show_waterfall


    
% ===============
% SHOW HISTOGRAMS
% ===============
if show_hist && filt_SCN
     
    % - User information -
    disp('Processing histograms...');

    % - Prepare graph space -
    scrsz  = get(0,'ScreenSize');  
    graph  = figure('Position',scrsz,'Color','white');    
    Axe{1} = [.09  .16  .35  .6];
    Axe{2} = [.5   .16  .35  .6];

    % - Left histogram -
    axes('position',Axe{1},'FontSize',fontsz); %graph subspace allowing white margin
    set(gca,'FontSize',fontsz,'FontWeight','normal');
    hold on; grid on; grid minor; box on;

    % - Annotation positions & PSD scale -
    median_x        = -158;
    median_y_filt   = 0.0110;
    median_y_nofilt = 0.0104;

    mean_y        = 0.008;
    mean_x_filt   = -161;
    mean_x_nofilt = -128;

    PSD_min = -180; %[dBW/Hz]
    PSD_max = -100; %[dBW/Hz]
    
    %---------------------------------%
    %  Plot histogram without filter  %
    %---------------------------------%

    % - User information -
    disp('Calculate histogram 1...');

    % - Calculate values for logaritmic histogram -
    binsize = 0.1;       %[dB] histogram bin size
    Hbins   = -200: binsize: 0;
    H1      = hist(PSD,Hbins);  
    H1      = H1/sum(H1);
    H1(1)   = 0;         %to avoid scaling on accumulated high negatives

    % - Plot histogam of wideband PSD samples -
    plot(Hbins,H1,'Color','red','LineWidth',1.5,'LineStyle','-');
    hold on;

    % - Calculate mean (RMS) and median PSD -
    psd = 10.^(PSD/10);
    psd_mean   = mean(  psd,'all','omitnan');
    psd_median = median(psd,'all','omitnan');
    PSD_mean   = 10*log10(psd_mean);   %[dBr]
    PSD_median = 10*log10(psd_median); %[dBr]

    % - Plot mean and median PSD -
    line(PSD_median*[1 1], [0 0.0015],'Color','black','LineWidth',2,'LineStyle','-');
    line(PSD_mean*  [1 1], [0 0.0015],'Color','black','LineWidth',2,'LineStyle','-');
    text(PSD_median, 0.002,'median','FontSize',fontsz*0.8,'Color','black','HorizontalAlignment','center');
    text(PSD_mean,   0.002,'mean',  'FontSize',fontsz*0.8,'Color','black','HorizontalAlignment','center');

    % - Graph make-up -
    title('Unfiltered','FontWeight','normal');
    ylabel('Occurrence \rightarrow','FontSize',fontsz);
    xlabel('PSD [dBW/Hz] \rightarrow',   'FontSize',fontsz);
    grid on;
    set(gca,'Fontsize',fontsz);
    axis([PSD_min PSD_max 0 max(H1) ]);
    yticks([]);

    %------------------------------%
    %  Plot histogram with filter  %
    %------------------------------%

    % - User information -
    disp('Calculate histogram 2...');

    % - Remove all blanked time-frequency bins -
    Pos          = find(PSD_SCN_filtered ~= BlankValue);
    PSD_SCN_filtered = PSD_SCN_filtered(Pos);

    % - Calculate values for logaritmic histogram -
    binsize = 0.1;       %[dB] histogram bin size
    Hbins   = -300: binsize: -30;
    H2 = hist(PSD_SCN_filtered,Hbins);  
    H2 = H2/sum(H2);

    % - Calculate mode PSD -
    Pos = find(H2==max(H2));
    P_peak   = Hbins(Pos);  %[dBr]

    % - Right histogram -
    axes('position',Axe{2},'FontSize',fontsz); %graph subspace allowing white margin
    set(gca,'FontSize',fontsz,'FontWeight','normal');
    hold on; grid on; grid minor; box on;

    % - Plot histogam of wideband PSD samples -
    plot(Hbins,H2,'Color','blue','LineWidth',1.5,'LineStyle','-');
    hold on;
    box on;

    % - Calculate mean (RMS) and median PSD -
    psd = 10.^(PSD_SCN_filtered/10);
    psd_mean   = mean(  psd,'all','omitnan');
    psd_median = median(psd,'all','omitnan');
    PSD_mean   = 10*log10(psd_mean);   %[dBr]
    PSD_median = 10*log10(psd_median); %[dBr]
    disp(' ');
    disp(['Difference between median and mean is ' num2str(round(PSD_mean-PSD_median,1)) ' dB.']);

    % - Plot mean and median PSD -
    line(PSD_median*[1 1], [0 0.0015],'Color','black','LineWidth',2,'LineStyle','-');
    line(PSD_mean*  [1 1], [0 0.0015],'Color','black','LineWidth',2,'LineStyle','-');
    text(PSD_median, 0.002,'median','FontSize',fontsz*0.8,'Color','black','HorizontalAlignment','right');
    text(PSD_mean,   0.002,'mean',  'FontSize',fontsz*0.8,'Color','black','HorizontalAlignment','left');

    % - Graph make-up -
    title( 'Filtered','FontWeight','normal');
    ylabel('Occurrence \rightarrow','FontSize',fontsz);
    xlabel('PSD [dBW/Hz] \rightarrow',   'FontSize',fontsz);
    grid on;
    set(gca,'Fontsize',fontsz);
    axis([PSD_min PSD_max 0 max(H2) ]);
    yticks([]);
       
end %show_hist

disp('  ');
disp('Patience... MatLab takes some time for detailed graphs..');
