import numpy as np
import matplotlib.pyplot as plt
from matplotlib import cm
import cblind as cb
from functions import fixPlot
#
# USER INPUT PARAMETERS
# 
save_Fig = True                         # Save the Figure prompt
show_Fig = True                         # Show the Figure prompt
figname = 'figures/figure2_a.pdf'         # Name of Figure
# Flow properties
nphases = 10                            # Number of wave phases
Ub = 0.072360125                        # Wave orbital velocity in m/s
Tw = 6.0                                # Wave Period in s
kvisc = 1.0e-6                          # Kinematic viscosity in m^2/s
kappa = 0.41                            # von Karman constant [-]
kc = 0.035                              # Height of the roughness in m
omega = 2*np.pi/Tw                      # Wave Frequency in rad/s
gamma = Ub*kc/omega                     # Relative roughness [-]
Rebk = Ub*kc/kvisc                      # Roughness Reynolds Number [-]
fac = kappa*kvisc/Ub**4                 # Dissipation normalisation factor
# 
# LOAD DATA
#
z = np.loadtxt('data/zgrid.dat')
# Load dissipation of TKE
serCy_epsilon = np.loadtxt('data/serialCylinder_epsilon.dat')
staCy_epsilon = np.loadtxt('data/staggeredCylinder_epsilon.dat')
serMa_epsilon = np.loadtxt('data/serialMassive_epsilon.dat')
staMa_epsilon = np.loadtxt('data/staggeredMassive_epsilon.dat')
serBr_epsilon = np.loadtxt('data/serialBranching_epsilon.dat')
staBr_epsilon = np.loadtxt('data/staggeredBranching_epsilon.dat')
morph1_epsilon = np.loadtxt('data/morph1_epsilon.dat')
morph2_epsilon = np.loadtxt('data/morph2_epsilon.dat')
#
# PLOT
#
color, linestyle = cb.Colorplots().cblind(12)
# Define the subplot type and size
fixPlot(thickness=1.5, fontsize=20, markersize=8, labelsize=15, texuse=True, tickSize = 15)
plt.figure(1,figsize=(8,8))
# Cylinder
plt.plot(np.nanmean(serCy_epsilon,axis=1)/fac,z/z[-1],'-x',label='C-Ser',color=color[0],markevery=10)
plt.plot(np.nanmean(staCy_epsilon,axis=1)/fac,z/z[-1],'-o',label='C-Stag',color=color[1],markevery=20)
# Massive
plt.plot(np.nanmean(serMa_epsilon,axis=1)/fac,z/z[-1],'-x',label='M-Ser',color=color[2],markevery=10)
plt.plot(np.nanmean(staMa_epsilon,axis=1)/fac,z/z[-1],'-o',label='M-Stag',color=color[3],markevery=20)
# Branching
plt.plot(np.nanmean(serBr_epsilon,axis=1)/fac,z/z[-1],'-x',label='B-Ser',color=color[4],markevery=10)
plt.plot(np.nanmean(staBr_epsilon,axis=1)/fac,z/z[-1],'-o',label='B-Stag',color=color[5],markevery=20)
# Morphology
plt.plot(np.nanmean(morph1_epsilon,axis=1)/fac,z/z[-1],'-x',label='Morph-1',color=color[8],markevery=10)
plt.plot(np.nanmean(morph2_epsilon,axis=1)/fac,z/z[-1],'-o',label='Morph-2',color=color[9],markevery=20)
# FORMATTING
plt.xlabel(r'$\langle \overline{\epsilon} \rangle^+ \equiv \langle \overline{\epsilon} \rangle \left( \frac{\kappa \nu}{U_b^4} \right) $',fontsize=20)
plt.ylabel(r'$x_3/H$',fontsize=20)
plt.grid()
plt.ylim([0,1.0])
plt.axhline(kc/z[-1],linewidth=2.0,linestyle=':',color='r')
plt.legend(frameon=False,fontsize=15,ncols=4)
# Set tight layout
plt.tight_layout()
## SAVE FIG
if(save_Fig):
    plt.savefig(figname,dpi=800)
if(show_Fig):
    plt.show()
