/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2016-2018 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "NASTRANMeshWriter.H"
#include "Time.H"
#include "Map.H"
#include "OFstream.H"
#include "processorPolyPatch.H"

#include "cellShape.H"
#include "hexMatcher.H"
#include "prismMatcher.H"
#include "tetMatcher.H"
#include "pyrMatcher.H"


#include <iostream>
#include <ios>
#include <sstream>
#include <iomanip>

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

bool Foam::fileFormats::NASTRANMeshWriter::binary         = false;
bool Foam::fileFormats::NASTRANMeshWriter::compress       = false;
bool Foam::fileFormats::NASTRANMeshWriter::prefixBoundary = true;

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

bool Foam::fileFormats::NASTRANMeshWriter::writeHeader(std::ofstream& os) const
{
	Info<< "Writing Header" << endl;
	
    os  << "$ FOAM to Nastran Mesh File" << std::endl
        << "BEGIN BULK" << nl
        << "$ Grid data section" << std::endl;

	
	return os.good();
}

bool Foam::fileFormats::NASTRANMeshWriter::writeGeometry(std::ofstream& os) const
{
    const pointField& points = mesh_.points();

	os.setf(std::ios::dec, std::ios::basefield);
	os.setf(std::ios::floatfield);
	os.precision(std::numeric_limits<double>::digits10 + 1);

	// Writing coordinates of vertices
    forAll(points, pointI)
    {
		const point& p = points[pointI];
		
        os
			<< std::left << std::left << std::setw(8) << "GRID*"
			<< std::right << std::setw(16) << pointI+1
			<< std::setw(16) << ""
			<< std::right << std::setw(16) << std::setprecision(13) << std::fixed << p.x()
			<< std::right << std::setw(16) << std::setprecision(13) << std::fixed << p.y() 
			<< std::left << std::setw(8) <<"*CONT" << std::endl;
		os	<< std::left << std::setw(8) <<"*CONT"
			<< std::right << std::setw(16) << std::setprecision(13) << std::fixed << p.z() 
			<< std::left << std::setw(16) << ""  << std::endl;
        }

    os << std::endl; // readability

    return os.good();
}


bool Foam::fileFormats::NASTRANMeshWriter::writeElements(std::ofstream& os) const
{
	// Writing cells
    os << "$ Element data section" << std::endl;

    const cellList& cells = mesh_.cells();
	
	labelList cellPts;
	
	hexMatcher hex;
        prismMatcher Pris;
        pyrMatcher Pyr;
        tetMatcher Tet;
	
	forAll(cells,cellI)
	{
		cellShape cs;
		
        if( hex.matches(mesh_, cellI, cs) )
        {
			os << std::left  << std::setw(8) << "CHEXA";
			os << std::right << std::setw(8) << cellI+1;
			os << std::right << std::setw(8) << "1";

			forAll(cs, i)
			{
                if( i == 2 )
                {
					//os << std::right << std::setw(8) << cs[i+1]+1;
					os << std::right << std::setw(8) << cs[i]+1;
                }
				// Comment: loop above and loop below are for swaping cell points with index 3 and 4.
				// This is the format for Nastran mesh imported in COMSOL as per the COMSOL document.
				// Similarly loops are inserted for swaping cell points with index 7 and 8.
                else if( i == 3 )
                {
					//os << std::right << std::setw(8) << cs[i-1]+1;
					os << std::right << std::setw(8) << cs[i]+1;
                }
                else if( i == 5 )
                {
					os << std::right << std::setw(8) << cs[i]+1 << std::left << std::setw(8) << "+CONT" << std::endl;
                }
                else if( i == 6 )
                {
					//os << std::left  << std::setw(8) << "+CONT" << std::right << std::setw(8)<< cs[i+1]+1;
					os << std::left  << std::setw(8) << "+CONT" << std::right << std::setw(8)<< cs[i]+1;
                }
                else if( i == 7 )
                {
					//os << std::right << std::setw(8) << cs[i-1]+1;
					os << std::right << std::setw(8) << cs[i]+1;
                }
                else
                {
					os << std::right << std::setw(8) << cs[i]+1;
                }
			}
                
			os << std::endl;
		}

         //

         if( Pris.matches(mesh_, cellI, cs) )
        {
                        os << std::left  << std::setw(8) << "CPENTA";
                        os << std::right << std::setw(8) << cellI+1;
                        os << std::right << std::setw(8) << "1";

                        forAll(cs, i)
                        {
                if( i == 2 )
                {
                                        //os << std::right << std::setw(8) << cs[i+1]+1;
                                        os << std::right << std::setw(8) << cs[i]+1;
                }
                                // Comment: loop above and loop below are for swaping cell points with index 3 and 4.
                                // This is the format for Nastran mesh imported in COMSOL as per the COMSOL document.
                                // Similarly loops are inserted for swaping cell points with index 7 and 8.
                else if( i == 3 )
                {
                                        //os << std::right << std::setw(8) << cs[i-1]+1;
                                        os << std::right << std::setw(8) << cs[i]+1;
                }
                else
                {
                                        os << std::right << std::setw(8) << cs[i]+1;
                }
                        }

                        os << std::endl;
                }
	//

        if( Pyr.matches(mesh_, cellI, cs) )
        {
                        os << std::left  << std::setw(8) << "CPYRAM";
                        os << std::right << std::setw(8) << cellI+1;
                        os << std::right << std::setw(8) << "1";

                        forAll(cs, i)
                        {
                if( i == 2 )
                {
                                        //os << std::right << std::setw(8) << cs[i+1]+1;
                                        os << std::right << std::setw(8) << cs[i]+1;
                }
                                // Comment: loop above and loop below are for swaping cell points with index 3 and 4.
                                // This is the format for Nastran mesh imported in COMSOL as per the COMSOL document.
                                // Similarly loops are inserted for swaping cell points with index 7 and 8.
                else if( i == 3 )
                {
                                        //os << std::right << std::setw(8) << cs[i-1]+1;
                                        os << std::right << std::setw(8) << cs[i]+1;
                }
                else
                {
                                        os << std::right << std::setw(8) << cs[i]+1;
                }
                        }

                        os << std::endl;
                }

     //

     if( Tet.matches(mesh_, cellI, cs) )
        {
                        os << std::left  << std::setw(8) << "CTETRA";
                        os << std::right << std::setw(8) << cellI+1;
                        os << std::right << std::setw(8) << "1";

                        forAll(cs, i)
                        {
                                        os << std::right << std::setw(8) << cs[i]+1;
                        }

                        os << std::endl;
                }

	}

    return os.good();
}

bool Foam::fileFormats::NASTRANMeshWriter::writeDataAndClose(std::ofstream& os) const
{
	os << "$ Property data section" << std::endl;
	os << "PSHELL         1" << std::endl;
	os << "ENDDATA" << std::endl;
	
	return os.good();
}

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::fileFormats::NASTRANMeshWriter::NASTRANMeshWriter
(
    const polyMesh& mesh,
    const scalar scaleFactor
)
:
    meshWriter(mesh, scaleFactor)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

bool Foam::fileFormats::NASTRANMeshWriter::write(const fileName& meshName) const
{
    bool useBinary   = binary;
    bool useCompress = compress;

    fileName baseName(meshName);
    if (baseName.empty())
    {
        baseName = meshWriter::defaultMeshName;

        const Time& t = mesh_.time();

        if
        (
            t.timeName() != "0"
         && t.timeName() != t.constant()
        )
        {
            baseName += "_" + t.timeName();
        }
    }
    else
    {
        baseName = baseName.lessExt();
    }
	
	Info << "baseName " << baseName << endl;

	fileName filename = baseName+".nas";

    autoPtr<std::ofstream> osPtr
    (
        new std::ofstream(filename.c_str())
    );

    if( osPtr->good() )
    {
        Info<< "Writing output to " << filename << endl;

		if( !writeHeader(osPtr()) )
		{
			Info << "Writting of points failed" << endl;
			
			std::exit(1);
		}
		
        if( !writeGeometry(osPtr()) )
		{
			Info << "Writting of vertices failed" << endl;
			
			std::exit(1);
		}
		
		if( !writeElements(osPtr()) )
		{
			Info << "Writting of elements failed" << endl;
			
			std::exit(1);
		}
		
		if( !writeDataAndClose(osPtr()) )
		{
			Info << "Cannot finalize writting anf close file" << endl;
			
			std::exit(1);
		}

        osPtr.clear();    // implicitly close the file
    }
    else
    {
        Info << "could not open file for writing " << filename << endl;
        
		return false;
    }

    return true;
}


// ************************************************************************* //
