#!/usr/bin/env python
import numpy as np
import rospy
import gym
import myenv
from timeit import default_timer as timer

from geometry_msgs.msg import TwistStamped, Twist, Pose

# from MPPI_vel import MPPI
from State_Estimator import State_Estimator
import sys
from controllers.mpc.free_push_mpc_controller_casadi import Free_Push_mpc_controller
from general_problem_setup import Problem, Index
from utils.utils import print_goal_info, fullState2generalState

import sys
from pathlib import Path

FILE_THIS = Path(__file__).resolve()
PARENT = FILE_THIS.parent
GPARENT = FILE_THIS.parents[1]

class Husky_Control:
    def __init__(self):
        self.pr = Problem()
        self.index = Index()

        # ros publisher for control
        self.pub_husky_control = rospy.Publisher("/husky_velocity_controller/cmd_vel", Twist, queue_size=1) 

        # Initializing robot mpc casadi
        self.mpc_controller = Free_Push_mpc_controller()
        recompile = True
        solver_build_name = 'free_push_mpc_controller_casadi'    
        self.mpc_controller.generate_solver(recompile, solver_build_name)

        self.obj_goal = self.pr.obj_pos_goal

    def get_null_twist(self):
        cmd = Twist()
        cmd.linear.x = 0.0
        cmd.linear.y = 0.0
        cmd.linear.z = 0.0
        cmd.angular.x = 0.0
        cmd.angular.y = 0.0
        cmd.angular.z = 0.0
        return cmd

    def pub_control_action(self, robot_control_current):
        cmd = self.get_null_twist()
        cmd.linear.x = robot_control_current[0]
        cmd.angular.z = robot_control_current[1]
        self.pub_husky_control.publish(cmd)


def run_control():
    print("Starting Control...")

    # Initialize a ros node
    rospy.init_node('mpc_control', anonymous=False)

    # Create a husky robot object
    husky = Husky_Control()

    # visualization
    vis_env = gym.make('cylinder_robot_pushing_recBox-v0')
    vis_env.init_path_candidates_both(1)

    # Create state estimation object 
    # mode = rospy.get_param('mode') # simulation or mocap
    mode = "mocap"
    state_estimator = State_Estimator(mode)
    # Get state from the state estimator
    full_state = state_estimator.get_state()
    system_state_current = fullState2generalState(full_state, husky.pr, husky.index)
    # show current state
    vis_env.set_state(system_state_current)
    # show current goal
    vis_env.init_goal_pos(husky.obj_goal)
    vis_env.render()

    # Specify the control node frequency
    hz = 10
    rate = rospy.Rate(hz)
    rospy.sleep(0.5)

    while not rospy.is_shutdown():
        # Get state from the state estimator
        full_state = state_estimator.get_state()
        system_state_current = fullState2generalState(full_state, husky.pr, husky.index)
        vis_env.set_state(system_state_current)

        # Set state to the controller
        husky.mpc_controller.set_system_state_current(system_state_current)
        # Set goal position
        husky.mpc_controller.set_obj_pos_theta_goal(husky.obj_goal, 0)
        # run mpc
        start = timer()
        husky.mpc_controller.run_mpc()
        end = timer()
        # print("Control Processing time: " + str(end - start))
        # retrieve info
        mpc_feasible = husky.mpc_controller.mpc_feasible
        action = husky.mpc_controller.robot_control_current
        # pub control signal
        husky.pub_control_action(action)
        print("Vel action: ", action)

        # visualization of the planned trajectory
        vis_env.show_paths_candidates_both(husky.mpc_controller.system_state_plan)
        vis_env.render()
        
        # Sleep until 0.1s done
        rate.sleep()

if __name__ == '__main__':
    rospy.sleep(1.0)

    run_control()
