clc; clear; close all;

%% DOCUMENTATION
% Author    : R. ten Hoopen
% Data      : Created 24-02-2020, last modified 15-11-2021
% Version   : 1.5

% This file extrapolates tail heavy off-road data using parametric and
% non-parametric methods on both time signal and rainflow matrices before
% implementing eight fatigue damage estimation methods.

% Dependencies function files
% - Load_data.m         (and preprocessing.m, and sniptool.m)
% - Threshold_selection.m
% - EVT.m               (and svvkernel.m)
% - PSD_extrapolation.m (and TimeseriesfromPSD.m)
% - create_PSD_envelope.m
% - RFM_to_load_spectra.m
% - limiting_RFM.m
% - cyclestodamage.m
% - PSDtoDamage.m


%% 1 INPUT

% Parameters
f_split         = 1/3;          % Split dataset into 1/3 training set and 2/3 validation set
Frac            = 0.2;          % Take fraction of training set [0.1, 0.2, 0.4 or 1]
N_EVT_ext       = 10;           % Desired POT extrapolation factor         
N_PSD_ext       = 10;           % Desired PSD extrapolation factor
N_samples       = 17;           % Choose number of samples (max 17)
m               =  6;           % choose between 5-8 i.e. material fatigue range from mil std 
E               =  193.4e9;     % Elasticity modulus

% Material properties
s_range = 0:0.1:2000;           % Define stress ranges
n_range = E*s_range.^-m;        % Calculate Wohler curve

% Plot wohler curve
figure(1)
plot(n_range,s_range);
axis([1e-2 1e12 1e0 s_range(end)+50])
set(gca, 'XScale', 'log')
% set(gca, 'YScale', 'log')
xlabel('Number of cycles to failure [-]')
ylabel('Stress [Mpa]')
axis([1e-1 max(n_range) 1 150])

%% 1.1 Load full dataset incl preprocessing

[xx,tt,ffs]    = Load_data(N_samples); 
%% 1.2 Split training and validation dataset

L = floor(cellfun('length',xx)*f_split);       % Define length of training set for each sample
for i = 1:length(L);                        
    X_training{i}   = xx{i}(1:floor(L(i)*Frac));   % Fraction from each sample is training data
    X_val{i}        = xx{i}(L(i):end);      % Rest is validation data 
end

% Combine separate fraction of samples
fs          =   mean(ffs);           % Add sample frequency 
x           =   cat(1, X_training{:});
x_val       =   cat(1, X_val{:});  % Make one time signal
t_val       =   linspace(0,(length(x_val)-1)/fs,length(x_val));


%% 2 Threshold selection and EVT for tail data

% Determine threshold graphically and verify
[u_MRL] = Threshold_selection(x,fs);        % Graphical way to determine threshold on data

%% 3 Select final threshold and apply EVT to fill tail of distribution
[x_ext,Exc_2] = EVT(x,fs,N_EVT_ext,u_MRL);


%Plot original and extrapolated signal
% section = (1950*fs:1960*fs);
% figure(23)
% plot(t(section),x(section),'b-')
% hold on
% for j=1:N_EVT_ext
%   T=section/fs;
%   xx=x_ext(section+(j-1)*length(x));
% plot(T,xx)
% hold on
% end
% title('Load history of an 8x8 military truck on various off-road sections')
% legend('Original','EVT')
% xlabel(['Time [s]'])
% ylabel(['Strain ', '[' , char(181),'m/m]'])
% axis([1950 1960 -350 350])

% figure(24)
% plot(t(116000:117000),x(116000:117000),'b-')%,t_ext(1:length(t)),x_ext(1:length(t)),'r:')
% title('Load history section an 8x8 military truck on various off-road sections')
% %legend('Original','EVT')
% xlabel(['Time [s]'])
% ylabel(['Strain ', '[' , char(181),'m/m]'])
% axis([t(116000) t(117000) -350 350])
%% 4 Calculate PSD from time history and generate statistically equivalent longer time history
clear Sxx_onesided; clear freq_onesided; clear x_PSD; clear t_PSD;

% Calculate PSD-plot of training sample and create extrapolated time series
   [Sxx_onesided, freq_onesided, x_PSD, t_PSD]     = PSD_extrapolation((x*E)/1e12, fs, N_PSD_ext);
   
%% 4.1 PSDs from EVT and VAL data
clear Sxx_onesided_VAL; clear freq_onesided_VAL;

% Calculate PSD-plot of validation data without extrapolation
[Sxx_onesided_VAL, freq_onesided_VAL,~,~]     = PSD_extrapolation((x_val*E)/1e12, fs, 0);

% Load PSD-envelope
[Sxx_envelop,freq_envelop] = create_PSD_envelope(freq_onesided_VAL,Sxx_onesided_VAL);

figure(40)
set(gcf,'color','w');
plot(freq_onesided,10*log10(Sxx_onesided),'k-',...
freq_onesided_VAL,10*log10(Sxx_onesided_VAL),'b-',...
freq_envelop,10*log10(Sxx_envelop),'m-')%,...
% freq_envelop2,10*log10(Sxx_envelop2),'r--',...
% freq_envelop3,10*log10(Sxx_envelop3),'k--')
legend('Original','POT','VAL','Envelop')
xlabel('f (Hz)')
ylabel('G_k(f) [dB/Hz]')
axis([0 100 -40 10])


%% 5 Perform rainflow count on all time histories
clear X

% Place all time histories in one cell variable
X{1} = x;                        % Original signal
X{2} = x_ext;                    % POT extrapolation
X{3} = (x_PSD*1e12)/E;           % PSD extrapolation and conversion to strain for comparison
X{4} = x_val;                    % Validation data

%% 6 Rainflow counting, goodman correction and normalization of load spectrum
clear S1; clear N1;

% Normalized load spectrum from time histories
[S1,N1] = RFM_to_load_spectra(X, fs); 
%% 6.1 Limiting RFM
clear S_lim; clear N_lim;

% Calculate limiting RFM from training sample
[S_lim, N_lim] = limiting_RFM([X{1}], fs);

% Store limiting RFM along with other S-N data
S1{end+1} = S_lim;
N1{end+1} = N_lim;
%% 7 Damage calculations time-domain:
clear D; clear D_tot;clear S;clear N;

[D,D_tot,S,N] = cyclestodamage(S1,N1,S_lim,N_lim,s_range,n_range,E); 

%% 8.1 Damage calculation frequency-domain

targetsize          = 5000;                 % Resample size for PSD (computational efficiency)
L                   = length(X{1})/fs;      % Training sample length
L1                  = length(X{4})/fs;      % Validation data length

% Damage, error and fatigue life from training sample
[D_tot_freq_x,E_rel_x,T_FL_x] = PSDtoDamage(Sxx_onesided,freq_onesided,targetsize,m,L,E,D_tot(1));

%Damage, error and fatigue life from validation
[D_tot_freq_x_EVT,E_rel_x_EVT,T_FL_x_EVT] = PSDtoDamage(Sxx_onesided_VAL,freq_onesided_VAL,targetsize,m,L1,E,D_tot(4)*N_EVT_ext)

%% 8.2 Envelope calculations

targetsize          = length(Sxx_envelop);           % Resample size for PSD (computational efficiency)

% Damage, error and fatigue life from envelope
[D_tot_freq_envelope,E_rel_envelope,T_FL_envelope] = PSDtoDamage(Sxx_envelop,freq_envelop,targetsize,m,T1,E,D_tot(1));























