function [Fixation_stats, Saccade_stats, x_medfilt, y_medfilt, p_medfilt,is_NaN] = fixation_filter(x,y,p)

% Fixation filter, tested for the Eyelink 1000 Plus.
% This function accepts gaze x- and y- coordinate vector in screen pixels, as well as a vector with the pupil Area (in Eyelink's arbitrary units)

% This function outputs filtered x, y, and pupil data vectors
% It also outputs a matrix 'Saccade_stats' containing the 1) start moment of the saccade (ms), 2) the end moment of the saccade (ms), and 3) the saccade amplitude (deg)
% Furthermore, it outputs a matrix 'Fixation_stats' containing (1) the start moment of the fixation (ms), (2) the end moment of the fixation (ms), (3) the fixation duration (ms), (4) the moment of fixation (ms), (5) the fixation x-coordinate (pixels), and 6) the fixation y-coordinate (pixels)

% The parameters listed below have been inspired from literature and adjusted through trial-and-error. This function has been tested for a recording at 2000 Hz.
% Nyström, M., & Holmqvist, K. (2010). An adaptive algorithm for fixation, saccade, and glissade detection in eyetracking data. Behavior Research Methods, 42, 188-204.

% The fixation filter is designed so that all missing data (such as due to blinks) are linearly interpolated. 
% Furthermore, the vector is partitioned into either fixation or saccade.
% This means that if adding the total fixation duration "sum(Fixation_stats(:,3))" and the total saccade duration "sum(Saccade_stats(:,2)-Saccade_stats(:,1))"
% , this would equal the length of the data vector (length(x)-1)/(tracker_Hz/1000)

% Caution: 
% Decide for yourself if the first and/or last fixation should be included in the computation of dependent variables such as the mean fixation duration.
% This would depend on the purpose of your experiment.
% The last fixation has typically not ended yet when the trial ends, so you may want to exclude it from the computation of mean fixation duration, i.e., "mean(Fixation_stats(1:end-1,3))" instead of "mean(Fixation_stats(:,3))"
% However, you may still want to use the last fixation location for e.g., scanpath analyses. 
% The same applies to blinks: it is up to you to decide whether fixations should be included or excluded if they overlap with blinks or other data gaps (i.e., is_NaN==1). 
%% Define parameters used for the fixation filter

eye_gap_margin = 100;                 % Amount of data to interpolate around NaN intervals for eye gaze, before and after (ms)
pupil_gap_margin = 100;               % Amount of data to interpolate around NaN intervals for pupil size, before and after (ms)
pupil_filt_interval = 50;             % Time interval of the moving median filter of the pupil size (ms)
pupil_unusual_rate = 4000;            % If the pupil size decreases more than this amount, then the eyelid is probably closing (a.u.)
median_filt_interval = 100;           % Time interval of the median filter (ms)
sg_length = 31;                       % Interval, or frame length, of the Savitzky-Golay filter (in samples)
sg_order = 2;                         % Order of the Savitzky-Golay filter
saccade_duration_min = 10;            % Minimum allowable saccade duration (ms)
saccade_duration_max = 150;           % Maximum allowable saccade duration (ms)
saccade_duration_expand = 10;         % Duration by which the saccade onset could be reduced and the saccade offset could be increased to find the local minimum of gaze speed (ms)
fixation_duration_min = 40;           % Minimum allowable fixation duration (ms)
pixels_horizontal = 1920;             % Number of pixels of screen, horizontal
pixels_vertical = 1080;               % Number of pixels of screen, vertical
p2mm = 0.2765625;                     % Pixel to mm conversion; 1 pixel is p2mm millimeters
distance_to_screen = 900;             % Distance between eyes and screen (mm)
tracker_Hz = 2000;                    % Recording frequency of the eye-tracker (Hz)
saccade_velocity_threshold = 30;      % Saccade velocity threshold (deg/s)
screen_pixel_margin = 50;             % Number of pixels outside screen for which the data are declared missing
produce_figures = 0;                  % If set to 1, then figures are created

if nargin < 3 % if no pupil size data is available
    p_medfilt = NaN;
end

%% Set out-of-screen gaze data to NaN
x(x<-screen_pixel_margin) = NaN;
x(x>pixels_horizontal+screen_pixel_margin) = NaN;
y(y>pixels_vertical+screen_pixel_margin) = NaN;
y(y<-screen_pixel_margin) = NaN;

%% Determine and set the gaps in the data

if nargin == 3 % if pupil size data is available
    p_movmean = movmean(p,pupil_filt_interval * (tracker_Hz/1000)); % The pupil Area is filtered using a moving mean filter.
    p_movmean_speed = tracker_Hz * diff(p_movmean);  % Pupil size change, in Area (a.u.) per second.
    unusual_pupil_change = find( p_movmean_speed < -pupil_unusual_rate); % if the pupil size is decreasing rapidly, then this is probably a semi-blink

    x(unusual_pupil_change)=NaN; % set to NaN
    y(unusual_pupil_change)=NaN; % set to NaN
    p(unusual_pupil_change)=NaN; % set to NaN
end

is_NaN=AddMarginAroundNaN(y,eye_gap_margin,tracker_Hz); % Obtain NaN values after adding a margin around each NaN

x_gaps=x;
x_gaps(is_NaN) = NaN; % x-gaze data with enlarged gaps
y_gaps=y;
y_gaps(is_NaN) = NaN; % y-gaze data with enlarged gaps

%% Convert pupil diameter to millimeters
if nargin == 3 % if pupil size data is available
    p = -2.4226161818*10^-7.*p.^2 + 2.3888590964*10^-3.*p + 1.4599981533; % Convert Eyelink's pupil Area to millimeters based on a previously established empirical fit (these parameters would need further checking based on your own setup
end
% Note: Eyelink converts Area (a.u.) to Diameter (a.u.) using 256 * sqrt(p / pi);
%% Linearly interpolate data gaps
x_gaps_filled = fixgaps(x_gaps); % linearly interpolate the NaNs. Use a constant value for leading and trailing NaNs.
y_gaps_filled = fixgaps(y_gaps); % linearly interpolate the NaNs. Use a constant value for leading and trailing NaNs.

if nargin == 3 % if pupil size data is available
    is_NaN=AddMarginAroundNaN(y,pupil_gap_margin,tracker_Hz); % Obtain NaN values after adding a margin around each NaN
    p_gaps = p;
    p_gaps(is_NaN) = NaN; % x-gaze data with enlarged gaps
    p_gaps_filled = fixgaps(p_gaps); % linearly interpolate the NaNs. Use a constant value for leading and trailing NaNs.
end

%% Filtering of the gaze signals

x_medfilt = movmedian(x_gaps_filled,median_filt_interval*(tracker_Hz/1000)+1);    % Apply a median filter to gaze x (add 1 to ensure the filter interval is odd, and hence symmetric)
y_medfilt = movmedian(y_gaps_filled,median_filt_interval*(tracker_Hz/1000)+1);    % Apply a median filter to gaze y (add 1 to ensure the filter interval is odd, and hence symmetric)
if nargin == 3 % if pupil size data is available
    p_medfilt = movmedian(p_gaps_filled,median_filt_interval*(tracker_Hz/1000)+1);    % Apply a median filter to pupil size (add 1 to ensure the filter interval is odd, and hence symmetric)
end

[x0,x1] = deal((x_medfilt-pixels_horizontal/2)*p2mm); % Convert x_medfilt coordinate to millimeters, and translated so that x = 0 corresponds to the horizontal center of the screen
[y0,y1] = deal((y_medfilt-pixels_vertical/2)*p2mm);   % Convert y_medfilt coordinate to millimeters, and translated so that y = 0 corresponds to the vertical center of the screen

x0(1,:) = []; % Remove one gaze point to ensure that x0 and x1, and y0 and y1 are shifted one index from each other
x1(end,:) = [];
y0(1,:) = [];
y1(end,:) = [];

GazeSpeed = [NaN;tracker_Hz*compute_angle(x0,y0,x1,y1,distance_to_screen)]; % Compute angular speed of target (deg/s). Add NaN in the beginning so that the length of the vector equals the time vector.

GazeSpeedFiltered=sgolayfilt(GazeSpeed,sg_order,sg_length); % Filter the angular speed using a Savitzky-Golay filtering

%% Extract saccades from the Gaze speed signal

is_Saccade=GazeSpeedFiltered > saccade_velocity_threshold; % Vector that indicates whether the eye-data is a saccade (1) or not (0). Here, a saccade is defined as a moment where the gaze speed exceeds a threshold angular speed (deg/s)
saccade_start_indices = find(is_Saccade(1:end-1) == 0 & is_Saccade(2:end) == 1); % Indices of starts of saccade_start_indices
saccade_end_indices   = find(is_Saccade(1:end-1) == 1 & is_Saccade(2:end) == 0); % Indices of ends of saccade_start_indices

if length(saccade_start_indices) == length(saccade_end_indices) % If the number of 'saccade starts' equals the number of 'saccade ends'
    SaccadeDuration=saccade_end_indices-saccade_start_indices; % Compute saccade duration
elseif length(saccade_start_indices) == length(saccade_end_indices)+1 % If there are more saccade starts than saccade ends (this occurs when the data vector ends in the middle of a saccade, and the saccade has not yet ended)
    SaccadeDuration=saccade_end_indices-saccade_start_indices(1:end-1); % Compute saccade duration; the last saccade is discarded
else
    error('Unexpected number of saccade starts and ends'); % A situation with more saccade ends than saccade starts should never occur (because of the preceding filtering).
end

Valid_saccade_indices = find(SaccadeDuration >= saccade_duration_min*(tracker_Hz/1000) & SaccadeDuration <= saccade_duration_max*(tracker_Hz/1000)); % Only retain saccades which are between the minimum and maximum permissable duration
saccade_start_indices = saccade_start_indices(Valid_saccade_indices); % Retain saccades
saccade_end_indices = saccade_end_indices(Valid_saccade_indices); % Retain saccades

%% Determine the 'true' start and end moments of the saccades. The true start and end moments are when the gaze speed becomes positive or negative, respectively

for i = 1:length(saccade_start_indices) % Loop over all identified saccades
    ei = saccade_end_indices(i)+saccade_duration_expand*(tracker_Hz/1000); % Index of the end of the saccade plus a small offset
    if ei > length(GazeSpeedFiltered)
        ei = length(GazeSpeedFiltered); % Constrain ei to the length of the data vector
    end
    saccade_end_index = saccade_end_indices(i) + find(diff(GazeSpeedFiltered(saccade_end_indices(i):ei))>0,1,'first')-2;
    if ~isempty(saccade_end_index)
        saccade_end_indices(i) = saccade_end_index; % Look for the first sign reversal of the filtered Gaze speed. This defines the true saccade end moment.
    else
        saccade_end_indices(i) = length(GazeSpeedFiltered);
    end

    si = saccade_start_indices(i)-saccade_duration_expand*(tracker_Hz/1000); % Index of the start of the saccade minus a small offset
    if si < 1
        si = 1; % Constrain si to the beginning of the data vector
    end

    saccade_start_index = saccade_start_indices(i) - find(diff(GazeSpeedFiltered(saccade_start_indices(i):-1:si))>0,1,'first')+2;
    if ~isempty(saccade_start_index)
        saccade_start_indices(i) = saccade_start_index; % Look for the first sign reversal of the filtered Gaze speed. This defines the true saccade start moment.
    else
        saccade_start_indices(i) = 1;
    end
end

%% Remove fixations shorter than the minimum fixation duration
if ~isempty(saccade_start_indices) % If there is at least one saccade

    if saccade_start_indices(1) < fixation_duration_min*(tracker_Hz/1000) % if the first saccade happens very shortly after the start of the trial (i.e., within the minimum fixation duration), then remove the first saccade
        saccade_start_indices = saccade_start_indices(2:end);
        saccade_end_indices = saccade_end_indices(2:end);
    end

    if length(saccade_start_indices) > 1 % If there are 2 or more saccades
        while 1 % Loop as long as there are no short fixations left

            ShortFixations = find(saccade_start_indices(2:end) - saccade_end_indices(1:end-1) < fixation_duration_min*(tracker_Hz/1000)); % Compute fixation durations, defined as the elapsed time between the end of a saccade and the start of the next saccade, and find indicees of fixations that are shorter than the minimum fixation duration

            if isempty(ShortFixations) % if there are no short fixations, then exist the loop
                break
            else % remove the fixation by removing the saccade start and end moments
                saccade_start_indices(ShortFixations+1) = [];
                saccade_end_indices(ShortFixations) = [];
            end
        end
    end
end
%% Create a matrix 'Saccade_stats' containing information about the saccades, and a matrix 'Fixation_stats' containing information about the fixations
if ~isempty(saccade_start_indices) % if there is at least one fixation

    x0=x_medfilt(saccade_start_indices); % Define start and end coordinates of the saccades. These are used to compute the saccade amplitude
    x1=x_medfilt(saccade_end_indices);
    y0=y_medfilt(saccade_start_indices);
    y1=y_medfilt(saccade_end_indices);

    x0 = (x0-pixels_horizontal/2)*p2mm; % Convert to millimeters, and translated so that x = 0 corresponds to the horizontal center of the screen
    x1 = (x1-pixels_horizontal/2)*p2mm; % Convert to millimeters, and translated so that x = 0 corresponds to the horizontal center of the screen
    y0 = (y0-pixels_vertical/2)*p2mm; % Convert to millimeters, and translated so that y = 0 corresponds to the vertical center of the screen
    y1 = (y1-pixels_vertical/2)*p2mm; % Convert to millimeters, and translated so that y = 0 corresponds to the vertical center of the screen

    Saccade_stats = [saccade_start_indices saccade_end_indices compute_angle(x0,y0,x1,y1,distance_to_screen)]; % Saccade start index, Saccade end index, Saccade amplitude (deg)

    Fixation_stats = NaN(size(saccade_start_indices,1)+1,6); % Allocate matrix for fixation stats. The number of fixations equals the number of saccades + 1

    Fixation_stats(:,1) = [1;saccade_end_indices(1:end)];                              % Index of start of fixation
    Fixation_stats(:,2) = [saccade_start_indices;length(GazeSpeed)];                   % Index of end of fixation
    Fixation_stats(:,3) = Fixation_stats(:,2)-Fixation_stats(:,1);                     % Duration of fixation
    Fixation_stats(:,4) = (Fixation_stats(:,1)+Fixation_stats(:,2))/2;                 % Moment of fixation

    for j=1:size(Fixation_stats,1) % loop over all fixations
        Fixation_stats(j,5) = mean(x_medfilt(Fixation_stats(j,1):Fixation_stats(j,2)));     % x-coordinate of fixation (pixels)
        Fixation_stats(j,6) = mean(y_medfilt(Fixation_stats(j,1):Fixation_stats(j,2)));     % y-coordinate of fixation (pixels)
    end

else % If there are no saccades at all, then treat the entire data vector as a fixation
    Saccade_stats = [NaN NaN NaN];
    Fixation_stats(1) = 1;
    Fixation_stats(2) = length(GazeSpeed);
    Fixation_stats(3) = Fixation_stats(2)-Fixation_stats(1);
    Fixation_stats(4) = (Fixation_stats(1)+Fixation_stats(2))/2;
    Fixation_stats(5) = mean(x_medfilt(Fixation_stats(1)+1:Fixation_stats(2)));
    Fixation_stats(6) = mean(y_medfilt(Fixation_stats(1)+1:Fixation_stats(2)));
end

Fixation_stats(:,1:4) = Fixation_stats(:,1:4)/(tracker_Hz/1000); % Convert indices to milliseconds
Saccade_stats(:,1:2) = Saccade_stats(:,1:2)/(tracker_Hz/1000); % Convert indices to milliseconds

if produce_figures == 1
    %% Linear gap interpolation for gaze x
    opengl hardware
    time = (1:length(x))/tracker_Hz; % Time vector in seconds
    figure;hold on;grid on
    plot(time,x,'k','Linewidth',3)
    plot(time,x_gaps_filled,'r','Linewidth',2)
    xlabel('Time (s)')
    ylabel('\it{x}\rm-coordinate (s)')
    set(gca,'xtick',0:1:13,'xlim',[0 13])
    fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
    set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
    %% Linear gap interpolation for gaze y
    figure;hold on;grid on
    plot(time,y,'k','Linewidth',3)
    plot(time,y_gaps_filled,'r','Linewidth',2)
    xlabel('Time (s)')
    ylabel('\it{y}\rm-coordinate (s)')
    set(gca,'xtick',0:1:13,'xlim',[0 13])
    fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
    set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
    %% Linear gap interpolation for pupil diameter
    if nargin == 3 % if pupil size data is available
        figure;hold on;grid on
        plot(time,p,'k','Linewidth',3)
        plot(time,p_gaps_filled,'r','Linewidth',2)
        xlabel('Time (s)')
        ylabel('Pupil diameter (mm)')
        set(gca,'xtick',0:1:13,'xlim',[0 13])
        fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
        set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
    end
    %% Plot gaze x-coordinate, color-coded as fixation (green) and saccade (red)

    figure;hold on;grid on
    for i=1:size(Fixation_stats)
        start_index=Fixation_stats(i,1)*(tracker_Hz/1000);
        end_index=Fixation_stats(i,2)*(tracker_Hz/1000);
        l(1)=plot(time(start_index:end_index),x_medfilt(start_index:end_index),'g','Linewidth',3);
    end
    for i=1:size(Saccade_stats)
        start_index=Saccade_stats(i,1)*(tracker_Hz/1000);
        end_index=Saccade_stats(i,2)*(tracker_Hz/1000);
        l(2)=plot(time(start_index:end_index),x_medfilt(start_index:end_index),'r','Linewidth',3);
    end
    xlabel('Time (s)')
    ylabel('\it{x}\rm-coordinate (s)')
    set(gca,'xtick',0:1:13,'xlim',[0 13])
    legend(l,'Fixation','Saccade','location','northwest')
    fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
    set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
    %% Plot gaze y-coordinate, color-coded as fixation (green) and saccade (red)
    figure;hold on;grid on
    for i=1:size(Fixation_stats)
        start_index=Fixation_stats(i,1)*(tracker_Hz/1000);
        end_index=Fixation_stats(i,2)*(tracker_Hz/1000);
        l(1)=plot(time(start_index:end_index),y_medfilt(start_index:end_index),'g','Linewidth',3);
    end
    for i=1:size(Saccade_stats)
        start_index=Saccade_stats(i,1)*(tracker_Hz/1000);
        end_index=Saccade_stats(i,2)*(tracker_Hz/1000);
        l(2)=plot(time(start_index:end_index),y_medfilt(start_index:end_index),'r','Linewidth',3);
    end
    xlabel('Time (s)')
    ylabel('\it{y}\rm-coordinate (s)')
    set(gca,'xtick',0:1:13,'xlim',[0 13])
    legend(l,'Fixation','Saccade','location','northwest')
    fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
    set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
    %% Plot the Savitzky-Golay-filted Gaze speed, with onsets and offsets of saccades
    figure;hold on;grid on
    plot(time,GazeSpeedFiltered,'k','Linewidth',3)
    for i=1:size(Saccade_stats)
        start_index=Saccade_stats(i,1)*(tracker_Hz/1000);
        end_index=Saccade_stats(i,2)*(tracker_Hz/1000);
        l(1)=plot(time(start_index),GazeSpeedFiltered(start_index),'mo','Linewidth',3,'Markersize',20);
        l(2)=plot(time(end_index),GazeSpeedFiltered(end_index),'co','Linewidth',3,'Markersize',20);
    end
    legend('Gaze speed','Saccade onset','Saccade offset','location','northwest')
    xlabel('Time (s)')
    ylabel('Speed of gaze point (deg/s)')
    fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
    set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
end
end
%% Function that computes the smallest possible rotation (in deg) that the eye has to perform to move from point (x0,y0,z0) to point (x1,y1,z1)
% Equation 4 from Cercenelli, L., Tiberi, G., Bortolani, B., Giannaccare, G., Fresina, M., Campos, E., & Marcelli, E. (2019). Gaze Trajectory Index (GTI): A novel metric to quantify saccade trajectory deviation using eye tracking. Computers in Biology and Medicine, 107, 86-96.

function angle = compute_angle(x0,y0,x1,y1,distance_to_screen)
z0=distance_to_screen;
z1=distance_to_screen;
angle=real(acosd((x0.*x1+y0.*y1+z0.*z1)./(sqrt(x0.^2+y0.^2+z0.^2).*sqrt(x1.^2+y1.^2+z1.^2))));
end

%% Find NaNs (blinks) in the data, and add a 'margin' number of NaNs surrounding each NaN transition. The function outputs a vector with 0s (no NaN) and 1s (NaN)
function is_NaN=AddMarginAroundNaN(y,gap_margin,tracker_Hz)

gap_margin = gap_margin * (tracker_Hz/1000); % convert gap margin from milliseconds to number of samples

is_NaN=isnan(y); % detect when there are NaNs in the eye-tracking data. NaNs may be caused by blinks.

NaN_onset_indices  = find(diff(is_NaN)==1); % indices of onset of NaN
NaN_offset_indices = 1+find(diff(is_NaN)==-1); % indices offset of NaN
NaN_switch_indices = unique([NaN_onset_indices;NaN_offset_indices]); % indexes where there is a switch from value to NaN or from NaN to value

% loop that adds an extra number of NaNs around each NaN transition. This is done because eye-tracking data quality is compromised during eyelid opening and closing.
for i=1:length(NaN_switch_indices)

    NaN_interval_with_margin = NaN_switch_indices(i) - gap_margin:NaN_switch_indices(i) + gap_margin; % interval of indices that should be set to NaN

    NaN_interval_with_margin(NaN_interval_with_margin < 1 ) = [];        % remove indexes smaller than 1
    NaN_interval_with_margin(NaN_interval_with_margin > length(y)) = []; % remove indexes larger than the data vector length

    is_NaN(NaN_interval_with_margin)=1; % set to NaN
end

end
%% Linearly interpolate missing data
function data = fixgaps(data)

if isnan(data(1)) % if the data vector starts with NaNs
    data(1:find(~isnan(data),1,'first')-1) = data(find(~isnan(data),1,'first')); % fill the NaNs with the first value in the vector
end

if isnan(data(end)) % if the data vector ends with NaNs
    data(find(~isnan(data),1,'last')+1:end) = data(find(~isnan(data),1,'last')); % fill the NaNs with the last value in the vector
end

NaN_indices=find(isnan(data)); % indices of NaNs
nNaN_indices=find(~isnan(data)); % indices of values (i.e., not NaN)

data(NaN_indices) = interp1(nNaN_indices,data(nNaN_indices),NaN_indices,'linear');

end