/*
 * Copyright (c) 2012 Crossing-Tech TM Switzerland. All right reserved.
 * Copyright (c) 2012, RiSD Laboratory, EPFL, Switzerland.
 *
 * Author: Simon Bliudze, Alina Zolotukhina, Anastasia Mavridou, and Radoslaw Szymanek
 * Date: 10/15/12
 */

package org.javabip.executor;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.javabip.api.*;
import org.javabip.exceptions.BIPException;

import java.lang.reflect.InvocationTargetException;
import java.time.LocalDateTime;
import java.util.*;

/**
 * The Kernel Executor which performs the execution of the corresponding BIP Component via its Behaviour. It is not a
 * multi-thread safe executor kernel therefore it should never be directly used. It needs to be proxied to protect it
 * from multi-thread access by for example Akka actor approach.
 *
 * At each execution cycle, the executor checks for the enabled internal, spontaneous and enforceable transition and
 * then either performs a transition or notifies the engine of the disabled ports.
 *
 * @author Alina Zolotukhina
 *
 */
public class ExecutorKernel extends SpecificationParser implements OrchestratedExecutor, ComponentProvider {

	String id;

	private BIPEngine engine;

	private ArrayList<String> notifiers = new ArrayList<String>();

	private ArrayList<Map<String, Object>> notifiersData = new ArrayList<Map<String, Object>>();

	protected boolean registered = false;

	protected static final Logger logger = LogManager.getLogger();
	//private Logger logger = LoggerFactory.getLogger(ExecutorKernel.class);

	private Map<String, Object> dataEvaluation = new Hashtable<String, Object>();

	boolean waitingForSpontaneous = false;

	protected OrchestratedExecutor proxy;

	/**
	 * By default, the Executor is created for a component with annotations. If you want to create the Executor for a
	 * component with behaviour, use another constructor
	 *
	 * @param bipComponent
	 *            the component to which the executor corresponds
	 * @param id
	 *            the executor id
	 * @throws BIPException
	 */
	public ExecutorKernel(Object bipComponent, String id) throws BIPException {
		this(bipComponent, id, true);
	}

	/**
	 * Creates a new executor instance.
	 *
	 * @param bipComponent
	 *            the BIP Component to which the executor corresponds
	 * @param id
	 *            the executor id
	 * @param useSpec
	 *            true, if the annotations are to be used, false otherwise
	 * @throws BIPException
	 */
	public ExecutorKernel(Object bipComponent, String id, boolean useSpec) throws BIPException {
		super(bipComponent, useSpec);
		this.id = id;
	}

	/*
	 * TODO, Proxy can be also obtained using this singleton proxy = TypedActor.<OrchestratedExecutor>self(); However,
	 * we tight ourselves to TypedActor singleton that can be disastrous in OSGi setup. However, at the same time any
	 * exception thrown from any function in this class will cause the TypedActor to die and a new one respawn making
	 * this proxy obsolete thus not being able to progress any further. Maybe, this is actually is not bad as we may
	 * want to have guarantees that after exception is being thrown no more function calls follow to this object.
	 */

	public void setProxy(OrchestratedExecutor proxy) {
		this.proxy = proxy;
		if (bipComponent instanceof BIPActorAware) {
			((BIPActorAware) bipComponent).setBIPActor(proxy);
		}
	}

	/**
	 * It registers the engine within given ExecutorKernel. The function setProxy must be called before with properly
	 * constructed proxy or an exception will occur.
	 */
	public void register(BIPEngine engine) throws BIPException {
		if (proxy == null) {
			throw new BIPException("Proxy to provide multi-thread safety was not provided.");
		}
		this.engine = engine;
		registered = true;
		waitingForSpontaneous = false;
		proxy.step();
	}

	public void deregister() {
		this.registered = false;
		this.waitingForSpontaneous = false;
		this.engine = null;
	}

	// Computed in guardToValue, used for checks in execute.
	private Map<String, Boolean> guardToValue;

	/**
	 *
	 * Defines one cycle step of the executor. If no engine is registered it will exit immediately.
	 *
	 * @return true if the next step can be immediately executed, false if a spontaneous event must happen to have
	 *         reason to execute next step again.
	 * @throws BIPException
	 */
	public void step() throws BIPException {
		// if the actor was deregistered then it no longer does any steps.
		if (!registered)
			return;

		dataEvaluation.clear();

		guardToValue = behaviour.computeGuardsWithoutData(behaviour.getCurrentState());

		//check invariant before a transition
		logger.debug("Invariant check at the beginning of each step {}", id);
		behaviour.checkInvariant();

		// we have to compute this in order to be able to raise an exception
		boolean existInternalTransition = behaviour.existEnabledInternal(guardToValue);

		if (existInternalTransition) {
			logger.debug("About to execute internal transition for component {}", id);

			behaviour.executeInternal(guardToValue);
			logger.debug("Issuing next step message for component {}", id);

			//informing engine
			engine.informInteral(proxy, behaviour.getCurrentState());

			// Scheduling the next execution step.
			proxy.step();
			logger.debug("Finishing current step that has executed an internal transition for component {}", id);

			//check invariant after transition
			logger.debug("Invariant check after an internal transition {}", id);
			behaviour.checkInvariant();

			return;
		}

		boolean existSpontaneousTransition = behaviour.existInCurrentStateAndEnabledSpontaneous(guardToValue);

		if (existSpontaneousTransition && !notifiers.isEmpty()) {

			for (int i = 0; i < notifiers.size(); i++) {

				String port = notifiers.get(i);

				if (behaviour.hasEnabledTransitionFromCurrentState(port, guardToValue)) {
					logger.debug("About to execute spontaneous transition {} for component {}", port, id);

					notifiers.remove(i);
					Map<String, ?> data = notifiersData.remove(i);

					// Both notifiers and notifiersData should be LinkedList to perform efficient removal from the
					// middle.
					if (data == null) {
						behaviour.executePort(port);
					} else {
						behaviour.execute(port, data);
					}

					//informing engine
					engine.informSpontaneous(proxy, behaviour.getCurrentState());
					logger.debug("Issuing next step message for component {}", id);

					// Scheduling the next execution step.
					proxy.step();
					logger.debug("Finishing current step that has executed a spontaneous transition for component {}",
							id);

					//check invariant after transition
					System.out.println("Invariant check after a spontaneous transition.");
					behaviour.checkInvariant();

					return;
				}
			}

		}

		boolean existEnforceableTransition = behaviour
				.existInCurrentStateAndEnabledEnforceableWithoutData(guardToValue)
				|| behaviour.existInCurrentStateAndEnforceableWithData();

		Set<Port> globallyDisabledPorts = behaviour
				.getGloballyDisabledEnforceablePortsWithoutDataTransfer(guardToValue);

		if (existEnforceableTransition) {
			logger.debug("About to execute engine inform for component {}", id);
			engine.inform(proxy, behaviour.getCurrentState(), globallyDisabledPorts);
			// Next step will be invoked upon finishing treatment of the message execute.

			//TODO not sure it is a good place, since the transition might be still not executed
			//check invariant after transition
			//System.out.println("Invariant check after an enforceable transition.");
			//checkInvariant();

			return;
		}

		/*
		 * existSpontaneous transition exists but spontaneous event has not happened yet, thus a follow step should be
		 * postponed until any spontaneous event is received.
		 * TODO: Tell the engine that I am waiting (send all disabled ports in the inform)
		 */
		if (existSpontaneousTransition) {
			logger.debug("Finishing current step for component {} doing nothing due no spontaneous events.", id);
			/*
			 * TODO for Natassa: Change the design of the engine to not continuously expect to be informed by all
			 * components even though there are no spontaneous transitions. So, if the component sends once that there
			 * are no transitions store this information and use it for the next cycles till you get something
			 * different. When this is done uncomment the next line.
			 */
			waitingForSpontaneous = true;
			// engine.inform(proxy, behaviour.getCurrentState(), globallyDisabledPorts);
			// Next step will be invoked upon receiving a spontaneous event.
			return;
		}

		// throw new BIPException("No transition of known type from state "
		// + behaviour.getCurrentState() + " in component "
		// + this.getId());

	}

	/**
	 * Executes a particular transition as told by the Engine
	 */
	public void execute(String portID) {

		if (portID != null) {

			if (dataEvaluation.isEmpty()) {

				if (behaviour.transitionNoDataGuardData(portID)) {
					behaviour.executePort(portID);
				}

				else if (!behaviour.existInCurrentStateAndEnabledEnforceableWithoutData(guardToValue))
					throw new BIPException("Port " + portID + " is not enabled in the current state");
				else {
					behaviour.executePort(portID);
				}
			} else {

				// Performing a check that all data provided make the transition enabled.
				List<Map<String, Object>> parameter = new ArrayList<Map<String, Object>>();
				parameter.add(dataEvaluation);

				try {
					if (!behaviour.checkEnabledness(portID, parameter).get(0)) {
						//throw new BIPException("Port " + portID+ " that requires data is not enabled for the received data");
					}
				} catch (Exception e) {
					System.out.println("start" + LocalDateTime.now());
					throw new BIPException(e);
				}

				behaviour.execute(portID, dataEvaluation);
			}
		}

		logger.debug("Issuing next step message for component {}", id);
		proxy.step();

	}

	public void inform(String portID) {
		inform(portID, null);
	}

	@Override
	public void inform(String portID, Map<String, Object> data) {

		// TODO DESIGN DISCUSSION what if the port (spontaneous) does not exist?. It should throw an exception or ignore it.
		if (portID == null || portID.isEmpty() || !behaviour.isSpontaneousPort(portID)) {
			return;
		}

		logger.info("{} was informed of a spontaneous transition {}", this.getId(), portID);

		notifiers.add(portID);
		notifiersData.add(data);

		if (waitingForSpontaneous) {
			logger.debug("Issuing next step message for component {}", id);
			waitingForSpontaneous = false;
			proxy.step();
		}
	}

	public <T> T getData(String name, Class<T> clazz) {

		if (name == null || name.isEmpty()) {
			throw new IllegalArgumentException("The name of the required data variable from the component "
					+ bipComponent.getClass().getName() + " cannot be null or empty.");
		}

		T result = null;

		try {
			logger.debug("Component {} providing data {}.", behaviour.getComponentType(), name);
			Object[] args = new Object[1];
			args[0] = bipComponent;
			Object methodResult = behaviour.getDataOutMapping().get(name).invokeWithArguments(args);

			if (!clazz.equals(Object.class) && !clazz.isAssignableFrom(methodResult.getClass())) {
				result = getPrimitiveData(name, methodResult, clazz);
			} else
				result = clazz.cast(methodResult);

		} catch (Throwable e) {
			ExceptionHelper.printExceptionTrace(logger.getMessageFactory(), e);
			e.printStackTrace();
		}
		return result;
	}

	Set<Class<?>> primitiveTypes = new HashSet<Class<?>>(Arrays.<Class<?>> asList(int.class, float.class, double.class,
			byte.class, long.class, short.class, boolean.class, char.class));

	<T> T getPrimitiveData(String name, Object methodResult, Class<T> clazz) {

		if (primitiveTypes.contains(clazz)) {

			/*
			 * For primitive types, as specified in primitiveTypes set, we use direct casting that will employ
			 * autoboxing feature from Java. Therefore, we suppress unchecked.
			 */
			@SuppressWarnings("unchecked")
			T result = (T) methodResult;

			return result;
		} else
			throw new IllegalArgumentException("The type " + methodResult.getClass()
					+ " of the required data variable " + name + " from the component "
					+ bipComponent.getClass().getName() + " does not correspond to the specified return type " + clazz);

	}

	public List<Boolean> checkEnabledness(PortBase port, List<Map<String, Object>> data) {
		try {
			return behaviour.checkEnabledness(port.getId(), data);
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (BIPException e) {
			e.printStackTrace();
		}
		return null;
	}

	public BIPComponent component() {
		if (proxy == null) {
			throw new BIPException("Proxy to provide multi-thread safety was not provided.");
		}
		return proxy;
	}

	public void setData(String dataName, Object data) {
		this.dataEvaluation.put(dataName, data);
	}

	public String toString() {
		StringBuilder result = new StringBuilder();

		result.append("BIPComponent=(");
		result.append("type = " + behaviour.getComponentType());
		result.append(", hashCode = " + this.hashCode());
		result.append(")");

		return result.toString();

	}

	public String getType() {
		return behaviour.getComponentType();
	}

	public Behaviour getBehavior() {
		return behaviour;
	}

	@Override
	public String getId() {
		return id;
	}

	@Override
	public BIPEngine engine() {
		return engine;
	}

	@Override
	public String getState() {
		return behaviour.getCurrentState();
	}

}