# -*- coding: utf-8 -*-
"""
@author: j.h.koo@tudelf.nl
"""

from pyomo.environ import (AbstractModel, Constraint, Reals, NonNegativeIntegers, Boolean, Param, Var,
                           NonNegativeReals, Objective, Set, minimize, Binary)

MT = 264  # the maximun discharge via turbines
MSP = int(11680)  # the maximum discharge via spillway gates
LWL = 60.0
FWL = 80.0
NHWL = 76.5
TWL_FF = 79.5
LWS = 385583000
FWS = 1408538000
NHWS = 1170402000
TWS_FF = 1337785000


def MPC_formula():
    model = AbstractModel()

    model.t = Set()
    model.QIN = Param(model.t)  # inflow input
    model.Md = Param(model.t)  # demand input
    model.I_RWS = Param()
    model.SO_P = Param(model.t)

    model.Z1 = Param()
    model.Z2 = Param()
    model.Z3 = Param()
    model.TWS_N = Param()
    model.TWS_L = Param()
    model.TWS_F = Param()

    model.SO = Var(model.t, domain=NonNegativeReals, bounds=(0, (MT + MSP)), initialize=model.SO_P)
    model.ST = Var(model.t, domain=NonNegativeReals, bounds=(0, MT), initialize=MT)
    model.SP = Var(model.t, domain=NonNegativeReals, bounds=(0, MSP), initialize=0)
    model.S = Var(model.t, domain=NonNegativeReals, bounds=(LWS, FWS), initialize=NHWS)
    model.d_S1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_S2 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_S3 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_ST = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_SO1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_SO2 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_SO3 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_SO4 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.max_SP = Var(within=NonNegativeReals)
    model.Obj_v = Var(model.t, within=NonNegativeReals, initialize=0)

    def _S1(model, t):
        if t == 0:
            return model.S[t] - model.I_RWS - (model.QIN[t] - model.SO[t]) * 3600 == 0
        return model.S[t] - model.S[t - 1] - (model.QIN[t] - model.SO[t]) * 3600 == 0
    model.storage1 = Constraint(model.t, rule=_S1)

    def _S2(model, t):
        return (LWS, model.S[t], FWS)
    model.storage2 = Constraint(model.t, rule=_S2)

    def SO_PT(model, t):
        return model.SO[t] - model.SP[t] - model.ST[t] == 0
    model.SO_PT = Constraint(model.t, rule=SO_PT)

    def ST_PT(model, t):
        return model.SO[t] - MT + model.d_ST[t] - model.SP[t] == 0
    model.ST_PT = Constraint(model.t, rule=ST_PT)

    def Demand_meet(model, t):
        return model.SO[t] - model.Md[t] >= 0
    model.demanddown = Constraint(model.t, rule=Demand_meet)

    def Stick_first_QTO(model, t):
        if t==0:
            return model.SO[0] - model.SO_P[1] == 0
        else:
            return model.SO[t] >= 0
    model.stick_first_qsp_ = Constraint(model.t, rule=Stick_first_QTO)

    def Gap_S_1(model, t):  # min model.d_S1[t]
        return model.S[t] - model.TWS_N - model.d_S1[t] <= 0
    model.Gap_S_1 = Constraint(model.t, rule=Gap_S_1)

    def Gap_S_2(model, t):  # min model.d_S2[t]
        return model.TWS_L - model.S[t] - model.d_S2[t] <= 0
    model.Gap_S_2 = Constraint(model.t, rule=Gap_S_2)

    def Gap_S_3(model, t):  # min model.d_S1[t]
        return model.S[t] - model.TWS_F - model.d_S3[t] <= 0
    model.Gap_S_3 = Constraint(model.t, rule=Gap_S_3)

    def SO_1(model, t):  # min model.d_QSP1[t]
        if t < 2:
            return model.d_SO1[t] + model.d_SO2[t] == 0
        else:
            return (model.SO[t-1] - model.SO_P[t]) / (int(t)-1) + model.d_SO1[t] - model.d_SO2[t] == 0
    model.SO_1 = Constraint(model.t, rule=SO_1)

    def SP_max(model, t):  # min model.max_QO
        return model.SP[t] - model.max_SP <= 0
    model.SP_max = Constraint(model.t, rule=SP_max)

    def Obj_var(model, t):
        return model.Obj_v[t] == (model.max_SP) * model.Z1 + \
                                 (model.d_S1[t] + model.d_S2[t] + model.d_S3[t]) * model.Z2 + \
                                 (model.d_SO1[t] + model.d_SO2[t]) * model.Z3 + \
                                 (model.d_ST[t]) * 0.01 * 1000 / MSP
    model.Obj_var = Constraint(model.t, rule=Obj_var)

    def obj_f(model):
        return sum(model.Obj_v[i] for i in model.t)
    model.obj = Objective(rule=obj_f, sense=minimize)

    return model
