% This model predicts the response of a thin plate immersed in fluid. The 
% modes are estimated with the function file "thinplatemodes". The model 
% file "model_main" is then run to predict the response of the gate. A
% measured wave impacts force is used as an example, loaded from the
% data folder.

%% Main case file
clear all; close all; clc;
addpath('../Model/');
addpath('../Model/Plotfunctions');
addpath('../Model/Forcefunctions');
addpath('../Model/Export_fig');

%% General settings
a.casecode = 'Case_name';
a.plotfolder = ['.\Results\' a.casecode '\']; if ~exist(a.plotfolder, 'dir'); mkdir(a.plotfolder); end
a.saveresults = 1;
a.saveplots = 1;

%% Case parameters
% Structural geometry parameters
v.Lx = 12; % Gate width [m]
v.Lz = 7.5; % Gate height [m]

% Structural strength/stiffness parameters
v.ths = 0.220; % Stiffness-equivalent solid gate thickness [m]
v.I = (1/12)*(v.ths^3);
v.thy = 0.5*v.ths; % Thickness between center of gravity and surface gate to determine stresses [m]
v.E = 200*10^9;
v.eta = 0.01;
v.EI = (1+v.eta*1j)*v.E*v.I;
v.nu = 0.3;
v.D = v.EI/(1-v.nu^2);
v.rhos = round(0.2*v.ths*7850);
v.thm = v.rhos/7850;
v.fy = 460*10^6; %Yield strength
v.cdamp = 0; %Distributed damping over entire plate surface (Ns/m)/m2

% Fluid geometry
v.f1.h = 7; % Water depth domain I
v.f1.Lx = v.Lx; % Sluice width domain II
v.f2.h = v.f1.h; % Water depth domain II
v.f2.Lx = 5*v.f1.Lx; % Sluice width domain II
v.f3.h = 4; % Water depth domain III
v.f3.Lx = v.f1.Lx; % Sluice width domain III

% Fluid constants
v.f1.rhof = 1025;
v.f2.rhof = 1025;
v.f3.rhof = 1025;
v.cp = 1500;
v.g = 9.81; v.f2.g = v.g; v.f1.g = v.g;

% Overhang length
v.f1.Ly = 3; % Length of domain I [m]

%% Accuracy settings
% Structure
cc.dx = 0.25; % distance between gridpoints (x-direction), determines accuracy of integrals, for fluid and structure
cc.dz = 0.25; % distance between gridpoints (z-direction), determines accuracy of integrals, for fluid and structure
cc.smodes = 5; %cc.xmodes*cc.zmodes;

% Fluid right 1
cc.f1.dx = cc.dx; cc.f1.dz = cc.dz; % must be equal to structural gridsize
cc.f1.fxmodes = 5; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f1.fzmodes = 5; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f1.modes = cc.f1.fxmodes*cc.f1.fzmodes;

% Fluid right 2
cc.f2.dx = cc.dx; cc.f3.dz = cc.dz; % must be equal to structural gridsize
cc.f2.fxmodes = 6; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f2.fzmodes = 6; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f2.modes = cc.f2.fxmodes*cc.f2.fzmodes;

% Fluid left
cc.f3.dx = cc.dx; cc.f2.dz = cc.dz; % must be equal to structural gridsize
cc.f3.fxmodes = 5; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f3.fzmodes = 5; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f3.modes = cc.f3.fxmodes*cc.f3.fzmodes;

% Fluid compressibility and surface wave (0=on/1=off)
a.waves = 0; % Include the effect of surface waves in the hydrodynamic response
a.compressible = 1; % Include the effect of compressibility in the hydrodynamic response

% Surface waves and compressibility settings
a.waves = 0; % Include the effect of surface waves in the hydrodynamic response
a.compressible = 1; % Include the effect of compressibility in the hydrodynamic response

%% Analysis settings
% Analysis type
a.analysistypes_all =  {'resonance_modes',... % Predicts the resonance modes of the gate in fluid (by peak picking)
                        'frequency_domain',... % Predicts the response in the frequency domain 
                        'time_domain'}; % Predicts the response in the time domain (by Fourier transforms)
a.analysistype = 'frequency_domain';

%% Structural modal shapes

%Solving thin plate equation to obtain modes
thinplatemodes(v,cc)

% Coordinate vectors structure (repeated in model, used for defining modes on grid here)
v.x = 0:cc.dx:v.Lx; 
v.z = 0:cc.dz:v.Lz; 

% Load mode solution and set for chosen grid size
modes = load(['.\Modes\omegan_size_' num2str(round(v.Lx*1000)) 'x' num2str(round(v.Lz*1000)) '_rho_' num2str(round(v.rhos)) '_EI_' num2str(round(real(v.EI))) '_eta_' num2str(v.eta) '.mat'],'omegan');
modes.Wkm = zeros(length(0:cc.dx:v.Lx),length(0:cc.dz:v.Lz),cc.smodes^2); %Preallocating
for k = 1:cc.smodes
    for m = 1:cc.smodes
    % Horizontal (analytical)
    betak(k) = k*pi/v.Lx;
    modes.Wx(:,k) = sin(betak(k).*v.x);
    % Vertical (numerically determined)
    load(['.\Modes\Wkm_size_' num2str(round(v.Lx*1000)) 'x' num2str(round(v.Lz*1000)) '_rho_' num2str(round(v.rhos)) '_EI_' num2str(round(real(v.EI))) '_eta_' num2str(v.eta) '_modes_' num2str(k) 'x' num2str(m) '.mat'],'sol')
    solz = real(deval(sol,v.z)); [~, i_max] = max(abs(solz(1,:)));
    modes.Wz(:,k,m) = solz(1,:)./solz(1,i_max); % normalising mode
    % Combined 2D modal shape
    modes.Wxz(:,:,k,m) = modes.Wx(:,k)*modes.Wz(:,k,m)';
    end
end
clear sol solz k m i_max betak

% % Plot of structural modal shapes
% if a.drysmodes.plot == 1; plot_drysmodes(v,a,cc,Wkm,omegan,a.drysmodes); end

%% Frequency-domain analysis
if strcmp(a.analysistype,'frequency_domain')

    % Force and frequency
    omegamin = 0; omegamax = 1000; opoints = 501; % Radial frequency range investigated
    f.omegavec = linspace(omegamin,omegamax,opoints);
    f = forceshape_round(v,cc,f,0.55,0.55,2,1); % Force amplitude shape
    
    % Output created during model run (for each frequency step, in this case only one)
    a.outputpar = {'w_poi'}; % Extra variable names to save can be added 'x','Wkm','max_pf_amp' for postprocessing
    a.poi = [6 7.5; 6 3; 1 3; 2 5]; % Points of interest [x z] on plate, where plate amplitudes and fluid pressures are evaluated (per frequency)
    a.pmotion = [1 0 0]; % 0/1 defl, 0/1 velocity, 0/1 acceleration
    a.pmotion_ylim = [0 (10^(-6)); 0 (2*10^-5); 0 (10^-2)]; % ylimits plots delf, vel, acc
    a.fmotion = [0 0]; % 0/1 pressure L, pressure R,
    %a.fmotion_ylim = [0 5; 0 5]; % ylimits plots defl [low high]; vel [low high]; acc
    a.pks = 0; % analyse natural frequencies of system in omega range by local peaks
    a.F = 0; % Plot force
    
    % Run frequency-domain model
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system for each coordinate

end

%% Find resonance modes
if strcmp(a.analysistype,'resonance_modes')
 
    % Set damping to zero (overwrite)
    v.cdamp = 0;
    v.eta = 0; v.EI = (1+v.eta*1j)*v.E*v.I; v.D = v.EI/(1-v.nu^2); modes.omegan = real(modes.omegan);  
        
    % Settings first iteration
    omegamin = 0; omegamax = 1000; opoints = 2001; % Radial frequency range investigated to find resonance frequencies
    f.omegavec = linspace(omegamin,omegamax,opoints);
    a.pks = 1; % analyse natural frequencies of system in omega range by local peaks
    f = forceshape_round(v,cc,f,0.55,0.55,2,1);
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients
    
    % Settings second iteration
    omegavec = [];
    for i = 1:length(out.omegapks)
        fvec = out.omegapks(i)-0.55:0.0025:out.omegapks(i)+0.55;
        omegavec = [omegavec fvec];
    end
    f.omegavec = omegavec;
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients
    
    % Result resonance frequencies
    tab_length = min(length(out.omegapks),cc.smodes^2);
    omegan_sorted = modes.omegan(:).';
    T = table((1:tab_length).',round(sort(omegan_sorted(1:tab_length)).'/(2*pi),1),round(out.omegapks(1:tab_length)/(2*pi),1).','VariableNames',{'No','Freq_dry','Freq_wet'}); % Result found res frequencies
    
    % Settings to plot orginal and resonance modes
    f.omegavec = out.omegapks;
    f = forceshape_round(v,cc,f,0.55,0.55,2,1);
    a.drysmodes.plot = 1; %plot, 0:off/1:2D /2: +contributions
    a.wetsmodes = struct('plot',1,'specificfolder','system/smodes_wet','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.pshape = struct('plot',0,'specificfolder','response/plate','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.fshapeL = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_L','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.fshapeR = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_R','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.fmodes = struct('plot',[0 0],'textfontsize',30); % 1: 1D, % 2: 2D
    a.plotfmodes = 5;
    a.fsliceL = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_L');
    a.fsliceR = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_R');
    
    % Run calculation for resonance frequencies only and plot output
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients
    fprintf('\n'); disp(T)
end

%% Time domain analysis
if strcmp(a.analysistype,'time_domain')   

    %% Process force signal to obtain input for model
    f = load('./Data/pressures_DH_s110_compl_long_units.mat'); % Data: p (pressures), z (location of measurements p), t (time vector)
    f = data_2Dv_changetimestep(f,0.001); % Sets desired timestep for time-domain analysis
    f = data_2Dv_addriseandfall(f,0.5,0.5); % adding linear rise and fall to pressure signals to start at zero
    f = data_2Dv_interpolate(v,cc,f,struct('bctop','closed')); % interpolate date to grid used for model calculations
    f.iv_x = [0:1:v.Lx]; % z-coordinates on borders intervals in which the force is divided
    f.iv_z = [0 2 4 5 6:0.25:7.5]; % z-coordinates on borders intervals in which the force is divided
    f.skewness_width = 0;
    f = data_3D_makeintervals(v,cc,f); % divide force into intervals with their own amplitude-time signals
    
    %% Obtain solution
    %% Output settings
    a.forceintervals = 1; % The time-domain signal is  divided over multiple intervals
    a.fft = 1; % Fourier-analysis, save solution for modal coefficients each frequency step
    a.poi = [6 7.5; 6 3; 1 3; 2 5]; % Points of interest [x z] on plate, where plate amplitudes and fluid pressures are evaluated (per frequency)
    a.outputpar = {'omegan','Wxz'}; % Extra variable names to save can be added 'x','Wkm','max_pfor_amp'
    
    %% Force and frequency (Fourier transform)
    maxomeg = 500*2*pi; % Limits the maximum frequency for which the response is taken into account 
    f.addstart = 1; % Adds time before signal (to see if response is zero before force 'starts'
    f.addend = 60; % Adds time after signal (to let the system damp out)
    f = model_fouriertransform(f,maxomeg);
    
    %% Run calculation of system response
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients
    Akm_H = out.Akm_H; clear out.Akm_H;
    
    %% Inverse Fourier transform
    fprintf('Performing inverse Fourier transform\n')
    
    % Time domain ifft
    n = length(f.interval(1).Y);
    if maxomeg > 0
        Akm_H(:,:,f.n_shortened+1:f.n_restore) = 0;
    end
    
    Akm_t = zeros(cc.smodes,cc.smodes,n);
    for k = 1:cc.smodes
        for m = 1:cc.smodes
            Hifft(1,:) = Akm_H(k,m,:);
            Yresp = Hifft;
            Yresptr(1:n/2+1) = Yresp;
            Yfliptr = fliplr(conj(Yresp)); Yresptr(n/2+2:n) = Yfliptr(2:end-1);
            Akm_t(k,m,:) = ifft(Yresptr);
        end
    end
    
    %% Postprocessing
    %% Reduce result to relevant time interval for output
    % Matrix for full after fourier transform time series is too large to
    % process (> 100 GB). Therefore, the relevant part of the interval is
    % extracted.
    f.t = f.t+f.tshift; % Set zero-point of time vector to original 
    t_min = -0.5; % Lower bound of extracted response interval
    t_max = 5.0; % Upper bound of extracted response interval
    [~,index_start] = find(f.t >= t_min,1,'first');
    [~,index_end] = find(f.t >= t_max,1,'first');
    
    ndt = 1;%round(dt_req/dt); % Reduce time step
    Akm_t = Akm_t(:,:,index_start:ndt:index_end);
    f.t = f.t(index_start:ndt:index_end);
    f.totalforce = f.totalforce(index_start:ndt:index_end);
    
    %% Structural modes to determine deflection output
    % The output grid size may differ from the one used in the calculation.
    cc.dx = 0.25;
    cc.dz = 0.25;
    v.x = 0:cc.dx:v.Lx; 
    v.z = 0:cc.dz:v.Lz;
    for k = 1:cc.smodes
        for m = 1:cc.smodes
        % Horizontal (analytical)
        betak(k) = k*pi/v.Lx;
        WxH(:,k) = sin(betak(k).*v.x);
        % Vertical (numerically determined)
        load(['.\Modes\Wkm_size_' num2str(round(v.Lx*1000)) 'x' num2str(round(v.Lz*1000)) '_rho_' num2str(round(v.rhos)) '_EI_' num2str(round(real(v.EI))) '_eta_' num2str(v.eta) '_modes_' num2str(k) 'x' num2str(m) '.mat'],'sol')
        solzH = real(deval(sol,v.z)); [~, i_max] = max(abs(solzH(1,:))); WzH(:,m) = solzH(1,:)./solzH(1,i_max); % normalising mode
        % 2D shape HxV
        Wkm(:,:,k,m) = WxH(:,k)*WzH(:,m)';
        end
    end
    
    %% Get plate deflection from modal coefficients (final solution)
    % The unit modal shapes are loaded forom the main gatefluid model. However,
    % a more close grid can also be used by evaluating the original BVP
    % solutions again.
    Wmodes = zeros(size(Wkm,1),size(Wkm,2),size(Akm_t,3),cc.smodes^2);
    fprintf('Composing plate response from structural coefficients\n')
    for i = 1:size(Akm_t,3)
        for k = 1:cc.smodes
            for m = 1:cc.smodes
                km = (k-1)*cc.smodes+m;
                Wmodes(:,:,i,km) = Akm_t(k,m,i)*Wkm(:,:,k,m);
            end
        end
    end
    wt = sum(Wmodes,4); % Full solution
    
    %% Location and time of maximum
    % Find location and time where maximum deflection occurs
    % Maximum absolute deflection per time step
    [max_wt(1,:)] = max(max(wt,[],1),[],2); max_wt_abs = abs(max_wt);
    [iit,it] = find(max_wt_abs == max(max_wt_abs));
    w_at_t_max(:,:) = wt(:,:,it);
    max_w_at_t_max = max(max(w_at_t_max));
    [iz,ix] = find(w_at_t_max == max_w_at_t_max,1,'first');
    
    xmax = v.x(ix);
    zmax = v.z(iz);
    tmax = f.t(it);
    wt_atmax = wt(iz,ix,:);
    
    %% Plate motion over time
    pmotionoptions = struct('cutoffcriterion','time','cutoffvalue',2,'plottype','withforce','specificfolder','dyn_t_resp','dataname','plateresponse','numaxis',1);
    [~] = plot_t_pmotion(wt,f,v,cc,a,pmotionoptions);
    plot_t_pcontributions(Akm_t,f,v,cc,a,pmotionoptions);
    
    %% Total deflection
    a.pshape = struct('plot',1,'dataname','Platedefl_wave_max','specificfolder','dyn_t_resp','title',0,'titlename','Deflection');
    plot_pshape(v,a,w_at_t_max,a.pshape)
    
    %% Stress at maximum deflection in time
    v.D = real(v.D); % Use real part of bending rigidity
    
    [FZ(:,:),FX(:,:)]=gradient(abs(w_at_t_max),v.z,v.x); %dw/dz(x,z) and dw/dx(x,z)
    FZ(:,end) = FZ(:,end-1)+(FZ(:,end-1)-FZ(:,end-2)) + ((FZ(:,end-1)-FZ(:,end-2))-(FZ(:,end-2)-FZ(:,end-3)));
    FZ(:,1) = FZ(:,2)+(FZ(:,2)-FZ(:,3)) + ((FZ(:,2)-FZ(:,3))-(FZ(:,3)-FZ(:,4)));
    [k_zz(:,:),k_xz(:,:)] = gradient(FZ,v.z,v.x); % dw^2/dz^2, dw^2/dxdz
    k_zz(:,end) = k_zz(:,end-1)+(k_zz(:,end-1)-k_zz(:,end-2)) + ((k_zz(:,end-1)-k_zz(:,end-2))-(k_zz(:,end-2)-k_zz(:,end-3)));
    k_zz(:,1) = k_zz(:,2)+(k_zz(:,2)-k_zz(:,3)) + ((k_zz(:,2)-k_zz(:,3))-(k_zz(:,3)-k_zz(:,4)));
    [~,k_xx(:,:)] = gradient(FX,v.z,v.x); % dw^2/dx^2
    
    sigma_xx = -v.D*v.thy/v.I*(k_xx + v.nu*k_zz);
    sigma_zz = -v.D*v.thy/v.I*(v.nu*k_xx + k_zz); 
    tau_xz = -v.D*v.thy/v.I*(1-v.nu)*(k_xz);
    
    sigma_v = sqrt(sigma_xx.^2 - sigma_xx.*sigma_zz + sigma_zz.^2 + 3*tau_xz.^2);
    sigma_v_max = max(max(abs(sigma_v)));
    
    %Plotting
    a.pstress = struct('calc',0,'plot',1,'dataname','Platestress_wave_max','specificfolder','dyn_t_resp','title',0,'titlename','von Mises stress','ylabelname','Von Mises stress [N/mm$^2$]');
    plot_pstress(v,a,sigma_v,a.pstress);
end