%% Case file
clear all; close all; clc; 
addpath('../Model/');
addpath('../Model/Forcefunctions');
addpath('../Model/Plotfunctions');
addpath('./SCIALink/');

%% General settings
a.casecode = 'Variable_gate';
a.plotfolder = ['.\Results\' a.casecode '\']; if ~exist(a.plotfolder, 'dir'); mkdir(a.plotfolder); end
a.modefolder = '.\Modes\'; if ~exist(a.modefolder, 'dir'); mkdir(a.modefolder); end

%% Analysis settings
% Analysis type
a.analysistypes_all =  {'static',...
                        'frequency_domain',...
                        'resonance_modes',...
                        'transfer_functions',...
                        'time_domain'};
a.analysistype = 'static';

% General settings 
a.loadmodes = 1;    % 1=loading modes if they are present
a.waves = 0;        % Include the effect of the surface waves in the hydrodynamic response
a.compressible = 1; % Include the effect of compressibility in the hydrodynamic response
a.saveresults = 0;
a.saveplots = 0;
a.stress = 1;       %Analyse stress


%% Case parameters

% Variable gate design parameters
v.Lx = 12; % Gate width [m]
v.Lz = 7.5; % Gate height [m]
v.case = 0; % Solid plate case - % Ask Ruben how this part works and decide what to do with it.
v.var = 1;
v.V(1,1) = 0.05; %plate thickness [m]
v.V(1,2) = 3.05;  %Widths(Intermediate stiffener 1)
v.V(1,4) = 6;     %Widths(Middle stiffener)
v.V(1,6) = 8.95;  %Widths(Intermediate stiffener 2)
v.V(1,3) = 2.55;  %Heights (Lower stiffener)
v.V(1,5) = 5.05;  %Heights (Higher stiffener)
v.V(1,7) = 0.1;   %Thickness (Edge vertical) [m]
v.V(1,8) = 0.1;   %Thickness (Intermediate vertical) [m]
v.V(1,9) = 0.1;   %Thickness (Middle vertical) [m]
v.V(1,10) = 0.1;  %Thickness (TopBottom horizontal) [m]
v.V(1,11) = 0.1;  %Thickness (Middle horizontal) [m]
v.V(1,12) = 1;    %Left constraint (1=rigid, 0=free)
v.V(1,13) = 1;    %Bottom constraint (1=rigid, 0=free)
v.V(1,14) = 1;    %Right constraint (1=rigid, 0=free)
v.V(1,15) = 0;    %Top constraint (1=rigid, 0=free)
v.V(1,16) = 0.5;  %Length horizontal ribs
v.V(1,17) = 0.5;  %Length vertical ribs
v.zbot = 0.0; % Sill height [m]

% Structural damping and strength parameters
v.eta = 0.01;
v.cdamp = 0; %Distributed damping over plate surface (Ns/m)/m2
v.fy = 460*10^6; %Yield strength

% Fluid geometry
v.f1.h = 7.5; % Water depth domain I
v.f1.Lx = v.Lx; % Sluice width domain II
v.f2.h = v.f1.h; % Water depth domain II
v.f2.Lx = v.f1.Lx; % Sluice width domain II
v.f3.h = 4; % Water depth domain III
v.f3.Lx = v.f1.Lx; % Sluice width domain III

% Fluid constants
v.f1.rhof = 1000;
v.f2.rhof = 1000;
v.f3.rhof = 1000;
v.cp = 1500;
v.g = 9.81; v.f2.g = v.g; v.f1.g = v.g;

% Overhang length
v.f1.Ly = 1; % Length of domain I [m]

%% Accuracy settings
% Structure
cc.dx = 0.05; % distance between gridpoints (x-direction), determines accuracy of integrals, for fluid and structure
cc.dz = 0.05; % distance between gridpoints (z-direction), determines accuracy of integrals, for fluid and structure
cc.smodes = 16; % total number of modes

% Fluid right 1
cc.f1.dx = cc.dx; cc.f1.dz = cc.dz; % must be equal to structural gridsize
cc.f1.fxmodes = 5; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f1.fzmodes = 5; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f1.modes = cc.f1.fxmodes*cc.f1.fzmodes;

% Fluid right 2
cc.f2.dx = cc.dx; cc.f3.dz = cc.dz; % must be equal to structural gridsize
cc.f2.fxmodes = 6;%cc.f1.fxmodes; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f2.fzmodes = 6; %cc.f1.fzmodes; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f2.modes = cc.f2.fxmodes*cc.f2.fzmodes;

% Fluid left
cc.f3.dx = cc.dx; cc.f2.dz = cc.dz; % must be equal to structural gridsize
cc.f3.fxmodes = 5;%cc.f1.fxmodes; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f3.fzmodes = 5; %cc.f1.fzmodes; % fluid modes truncated to (fxmodes x fzmodes) potential shapes
cc.f3.modes = cc.f3.fxmodes*cc.f3.fzmodes;

% Coordinate vectors structure (repeated in model, used for defining modes on grid here)
v.x = 0:cc.dx:v.Lx;
v.z = 0:cc.dz:v.Lz;

Optimization_parameters; % Loads the system parameters & fills in the current variables

%% Link with SCIA for modal shapes
if a.loadmodes == 1
    T = readtable('./Modes/frequencies.xlsx');
    modes.frequency = table2array(T);
    modes.omegan = (1+v.eta*1j)*2*pi*modes.frequency(1:cc.smodes).';
    path = ['.\Modes\verts.xlsx'];
    T = readtable(path);
    modes.verts = table2array(T);
    path = ['.\Modes\faces.xlsx'];
    T = readtable(path);
    modes.faces = table2array(T);
    path = ['.\Modes\coldisp.xlsx'];
    T = readtable(path);
    modes.col_disp = table2array(T);
    path = ['.\Modes\colstress.xlsx'];
    T = readtable(path);
    modes.col_stress = table2array(T);
    path = ['.\Modes\colstressneg.xlsx'];
    T = readtable(path);
    modes.col_stress_neg = table2array(T);
    path = ['.\Modes\coltmax.xlsx'];
    T = readtable(path);
    modes.col_tmax = table2array(T);
    for i = 1:cc.smodes
        path = ['.\Modes\mode' num2str(i) '.xlsx'];
        T = readtable(path);
        modes.Wxz(:,:,i) = table2array(T);
    end
else
    [modes] = Optimization_SCIA(v,cc);
    modes.omegan = (1+v.eta*1j)*2*pi*modes.frequency(1:cc.smodes).';
end

%% Static analysis
if strcmp(a.analysistype,'static')  
    
    % Frequency and force shape - e.g. hydrostatic pressure or distributed load
    f.omegavec = 0.001;
%     pf_R = zeros(round(v.Lx/cc.dx+1),round(v.Lz/cc.dz+1));
%     for i = 1:(v.f1.h/cc.dz+1)
%         pf_R(:,i) = v.f1.rhof*v.g*(v.f1.h-(i-1)*cc.dz);
%     end
%     f.F(:,:,1,1) = pf_R;    
    f.F(:,:,1,1) = 10000*ones(round(v.Lx/cc.dx+1),round(v.Lz/cc.dz+1),1,1); 
    plot_force(v.x,v.z,f,a)
    
    % Run frequency-domain model with desired output
    a.outputpar = {'w','Akm'}; %Sets which variables during model run are saved for output.
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system for each coordinate
    
    % Plot deflection (2D)
    a.pshape = struct('plot',1,'dataname','Plate_deflection','specificfolder','static','ylabelname','Deflection [mm]','title',1,'titlename','Static deflection','textfontsize',20);
    max_deflection = max(max(abs(out.w)));
    plot_pshape(v,a,abs(out.w),a.pshape)
    
%     for km = 1:cc.smodes
%         name = ['./Figures/1.0_mode' num2str(km) '.jpg'];
%         saveas(surf(modes.Wxz(:,:,km)), name);
%     end
    
    % Plot equivalent stress+ front plate (2D)
    if a.stress == 1
    % Plot deflection (3D)
    for km = 1:cc.smodes
        Wxz_modes(:,km) = out.Akm(km)*modes.col_disp(:,km);
    end
    defl3D = sum(Wxz_modes,2);
    a.pshape = struct('plot',1,'dataname','Plate_deflection_3D','specificfolder','static','ylabelname','Deflection [mm]','title',1,'titlename','Static deflection','textfontsize',20);
    plot_pshape3D(a,modes,defl3D,a.pshape)

    % Plot stress (3D)
    for km = 1:cc.smodes
        Stressmodes3D(:,km) = out.Akm(km)*modes.col_stress(:,km);
    end
    stress3D = sum(Stressmodes3D,2);
    a.pstress = struct('plot',1,'axis_normalised',1,'dataname','Plate_stress_3D','specificfolder','static','ylabelname','Stress [N/mm$^2$]','title',1,'titlename','Static stress','textfontsize',20);                    
    plot_pstress3D(a,modes,stress3D/10^6,a.pstress)
    
    for km = 1:cc.smodes
        Stressmodes3D(:,km) = out.Akm(km)*modes.col_tmax(:,km);
    end
    stress3D = sum(Stressmodes3D,2);
    a.pstress = struct('plot',1,'axis_normalised',1,'dataname','Plate_stress_3D','specificfolder','static','ylabelname','Stress [N/mm$^2$]','title',1,'titlename','Static stress','textfontsize',20);                    
    plot_pstress3D(a,modes,stress3D/10^6,a.pstress)
    end

end

%% Frequency-domain analysis
if strcmp(a.analysistype,'frequency_domain')
    % Force and frequency
    omegamin = 0; omegamax = 1000; opoints = 501; % Radial frequency range investigated
    f.omegavec = linspace(omegamin,omegamax,opoints);
    f = forceshape_round(v,cc,f,0.55,0.55,2,1); % Force amplitude shape

    % Output created during model run (for each frequency step, in this case only one)
    a.outputpar = {'w_poi'}; % Extra variable names to save can be added 'x','Wkm','max_pf_amp' for postprocessing
    a.poi = [0.1 0.1; 0.4 0; 0.4 0.3; 0.4 v.Lz; 0.7 0.5]; % Points of interest [x z] on plate, where plate amplitudes and fluid pressures are evaluated (per frequency)
    a.pmotion = [1 0 0]; % 0/1 defl, 0/1 velocity, 0/1 acceleration
    a.pmotion_ylim = [0 (5*10^(-7)); 0 (2*10^-5); 0 (10^-2)]; % ylimits plots delf, vel, acc
    a.fmotion = [0 0]; % 0/1 pressure L, pressure R,
    %a.fmotion_ylim = [0 5; 0 5]; % ylimits plots defl [low high]; vel [low high]; acc
    a.pks = 0; % analyse natural frequencies of system in omega range by local peaks
    a.F = 0; % Plot force

    % Run frequency-domain model
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system for each coordinate
end

%% Resonance mode analysis
if strcmp(a.analysistype,'resonance_modes')

    % Set damping to zero (overwrite)
    v.cdamp = 0;
    modes.omegan = real(modes.omegan);

    % Settings first iteration
    omegamin = 0; omegamax = 1000; opoints = 2001; % Radial frequency range investigated to find resonance frequencies
    f.omegavec = linspace(omegamin,omegamax,opoints);
    a.pks = 1; % analyse natural frequencies of system in omega range by local peaks
    f = forceshape_round(v,cc,f,0.55,0.55,2,1);
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients

    % Settings second iteration
    omegavec = [];
    for i = 1:length(out.omegapks)
        fvec = out.omegapks(i)-0.55:0.0025:out.omegapks(i)+0.55;
        omegavec = [omegavec fvec];
    end
    f.omegavec = omegavec;
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients

    % Result resonance frequencies
    tab_length = min(length(out.omegapks),cc.smodes);
    T = table((1:tab_length).',round(sort(modes.omegan(1:tab_length)).'/(2*pi),1),round(out.omegapks(1:tab_length)/(2*pi),1).','VariableNames',{'No','Freq_dry','Freq_wet'}); % Result found res frequencies

    % Settings to plot orginal and resonance modes
    f.omegavec = out.omegapks;
    a.pshape = struct('plot',0,'specificfolder','response/plate','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.fshapeL = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_L','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.fshapeR = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_R','axis_normalised',1,'nmodes_contr',5,'textfontsize',35); %plot, 0:off/1:2D /2: +contributions
    a.fmodes = struct('plot',[0 0],'textfontsize',30); % 1: 1D, % 2: 2D
    a.plotfmodes = 5;
    a.fsliceL = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_L');
    a.fsliceR = struct('plot',0,'title',1,'specificfolder','system/fmodes_wet_R');

    % Run calculation for resonance frequencies only and plot output
    a.outputpar = {'Akm_H'}; %Sets which variables during model run are saved for output.
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients
    fprintf('\n'); disp(T)

    % Plot of dry structural modal shapes (2D)
    a.drysmodes.plot = 0; 
    if a.drysmodes.plot == 1; plot_drysmodes(v,a,cc,modes.Wxz,modes.omegan,a.drysmodes); end
           
    % Plot dry stress modes (3D)
    a.drysmodes.plot = 0; 
    if a.stress == 1 && a.drysmodes.plot == 1
        for km = 1:cc.smodes
            a.pstress = struct('plot',1,'axis_normalised',1,'dataname',['Stress_mode_3D_' num2str(km)],'specificfolder','/system/smodes_dry','title',1,'titlename',['Dry stress mode ' num2str(km)]);
            plot_pstress3D(a,modes,modes.col_stress(:,km),a.pstress)
        end
    end
    
    % Obtain normalized wet resonance shapes deflection and stress (2D an)
    for ff = 1:size(out.Akm_H,2)
        for km = 1:size(out.Akm_H,1)
            resonance_defl_modes(:,:,km) = out.Akm_H(km,ff)*modes.Wxz(:,:,km);
            if a.stress==1; resonance_stress_modes(:,km) = out.Akm_H(km,ff)*modes.col_stress(:,km); end
        end        
        % Normalise defl shapes
        resonance_defl(:,:,ff) = sum(resonance_defl_modes,3);
        phasesign = real(sign(resonance_defl(:,:,ff)));
        phasesign(isnan(phasesign)==1) = 0;
        resonance_defl(:,:,ff) = abs(resonance_defl(:,:,ff)).*phasesign;
        mm_w_abs = max(max(abs(resonance_defl(:,:,ff))));
        [ii,jj] = find(abs(resonance_defl(:,:,ff)) == mm_w_abs,1,'first');
        mm_w_amp = resonance_defl(ii,jj,ff);
        resonance_defl(:,:,ff) = resonance_defl(:,:,ff)/mm_w_amp;
        
        % Normalise stress shapes
        if a.stress == 1
        resonance_stress(:,ff) = sum(resonance_stress_modes,2); 
        phasesign = real(sign(resonance_stress(:,ff)));
        phasesign(isnan(phasesign)==1) = 0;
        resonance_stress(:,ff) = abs(resonance_stress(:,ff)).*phasesign;
        mm_w_abs = max(max(abs(resonance_stress(:,ff))));
        [ii,~] = find(abs(resonance_stress(:,ff)) == mm_w_abs,1,'first');
        mm_w_amp = resonance_stress(ii,ff);
        resonance_stress(:,ff) = resonance_stress(:,ff)/mm_w_amp;
        end
    end
    
    % Plot resonance shapes wet deflections (2D)
    a.wetmodes.plot = 0; 
    if a.wetmodes.plot == 1
        for i = 1:size(resonance_defl,3)
        a.pshape = struct('plot',1,'axis_normalised',1,'dataname',['Resonance_shape_wet_' num2str(i)],'specificfolder','/system/smodes_wet','title',1,'titlename',['Resonance shape at $f_{' num2str(i) '} = ' num2str(round(f.omegavec(i)/(2*pi),1)) '$ Hz']);
        plot_pshape(v,a,resonance_defl(:,:,i),a.pshape);
        end
    end
    
    % Plot wet resonance stress shapes (3D)
    a.wetsmodes.plot = 1; 
    if a.stress == 1 && a.wetsmodes.plot == 1
        for ff = 1:size(out.Akm_H,2)
            a.pstress = struct('plot',1,'axis_normalised',1,'dataname',['Stress_resonance_shape_3D_' num2str(km)],'specificfolder','/system/smodes_wet','title',1,'titlename',['Resonance stress shape at $f_{' num2str(ff) '} = ' num2str(round(f.omegavec(ff)/(2*pi),1)) '$ Hz']);
            plot_pstress3D(a,modes,resonance_stress(:,ff),a.pstress)
        end
    end           
end

%% Transfer functions per interval
if strcmp(a.analysistype,'transfer_functions')
    %% Run model   
    % Frequency settings
    frequencymin = 0; frequencymax = 200; fpoints = 4001; % Frequency [Hz] range investigated
    f.transfer_freq = linspace(frequencymin,frequencymax,fpoints);
    f.transfer_omega = 2*pi*f.transfer_freq;
    f.omegavec = f.transfer_omega;
    
    % Interval settings
    f.iv_x = [0:3:v.Lx]; % z-coordinates on borders intervals in which the force is divided
    f.iv_z = [0:1.5:v.Lz]; % z-coordinates on borders intervals in which the force is divided

    f.ii_ivx = round(f.iv_x/cc.dx+1,0); % Indexes of interval boundaries in x-dir
    f.n_ivx = length(f.ii_ivx)-1; % Number of isntervals in x-dir
    f.ii_ivz = round(f.iv_z/cc.dz+1,0); % Indexes of interval boundaries in z-dir
    f.n_ivz = length(f.ii_ivz)-1; % Number of intervals in z-dir

    for ii = 1:f.n_ivx  % loop over x-intervals
        for jj=1:f.n_ivz % loop over z-intervals
            f.interval(ii,jj).F = zeros(length(0:cc.dx:v.f1.Lx),length(0:cc.dz:v.f1.h),1,1);
            f.interval(ii,jj).F(f.ii_ivx(ii):f.ii_ivx(ii+1),f.ii_ivz(jj):f.ii_ivz(jj+1),1,1) = 1;
        end
    end

    a.forceintervals = 1; % Set analysis to include force intervals
    a.nwaves = 1; % Set analysis to return individual RAO's instead of sum them within model
    a.outputpar = {'interval'}; % Extra variable or structure names to save can be added 'x','Wkm','max_pf_amp' for postprocessing
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system for each coordinate
    interval = out.interval;

    %% Plot response amplitude operator for an interval (as an example)
    plotexample = 0;
    if plotexample == 1
    figure
    ii = 1; jj = 2; % Chosing an interval to plot
    subplot(1,4,1:3)
    semilogy(f.transfer_freq,abs(interval(ii,jj).Aln_RAO(:,:))); hold on
    xlim([f.transfer_freq(1) f.transfer_freq(end)]); grid on;
    xlabel('Frequency [Hz]','Interpreter','Latex','Fontsize',10); ylabel('Modal coefficient [m]','Interpreter','Latex','Fontsize',10);
    set(gca,'Ticklabelinterpreter','Latex','Fontsize',12)
    for i = 1:cc.smodes; legendentries{i} = ['Mode ' num2str(i)]; end; legend(legendentries)
    title(['Frequency response function for interval (' num2str(ii) ',' num2str(jj) ') per mode'])
    subplot(1,4,4);
    rectangle('Position',[f.iv_x(ii) f.iv_z(jj) f.iv_x(ii+1)-f.iv_x(ii) f.iv_z(jj+1)-f.iv_z(jj)],'LineWidth',1,'Edgecolor','k','Facecolor',[0.6 0.6 0.6]); hold on
    box on; grid on; axis equal; xlim([0 v.Lx]); ylim([0 v.Lz]); xticks(f.iv_x); yticks(f.iv_z);
    xlabel('x-direction [m]','Interpreter','Latex','Fontsize',10); ylabel('z-direction [m]','Interpreter','Latex','Fontsize',10);
    set(gca,'Ticklabelinterpreter','Latex','Fontsize',12)
    title(['Location of interval (' num2str(ii) ',' num2str(jj) ')'])
    set(gcf, 'units','normalized','outerposition',[0.05 0.25 0.9 0.5],'color','white');
    end

    %% Save response function
    a.transferfolder = ['.\Results\' a.casecode '\transferfunctions\']; if ~exist(a.transferfolder, 'dir'); mkdir(a.transferfolder); end
    save([a.transferfolder '\hL_' num2str(v.f1.h*1000) 'mm_hR_' num2str(v.f3.h*1000) 'mm.mat'],'interval','f') % save transfer functions per interval
    if exist([a.transferfolder 'modes.mat'],'file') ~= 2; save([a.transferfolder '\modes.mat'],'modes','v'); end % save displacement and stress modes
end

%% Time-domain analysis (gate deflection)
if strcmp(a.analysistype,'time_domain')      
    %% Force input from Perigrine Cooker
    % Desired output
    dz      = cc.dz; % m
    dt      = 0.0005; % s
    t_end   = 1;  % s
    t = 0:dt:t_end;
    z = 0:dz:v.f1.h;
    
    % Wave parameters
    H_w = 0.08; % [m]
    T_w = 1.6; % [s]
    omega_w = (2*pi)/T_w; % [rad/s]
    tau = 0.010; % [s]
    cr = 1; % [-]
    U = (1+cr)*omega_w*(H_w/2);
    beta_mu = 1.17; % [-]This is
            
    % Wood and perigrine pressure impulse shape 
    wp_a = round((v.f1.h/v.f1.Ly)/cc.dz)*cc.dz; wp_dz = cc.dz/v.f1.Ly; W = v.f1.Ly;
    Pz_impact_U1 = v.f1.rhof*W*woodandperigrine(wp_a,wp_dz);
    
    % Velocity and resulting pressure impulses at zero-crossings 
    Pz_impact = Pz_impact_U1*U;
    pz_peak = beta_mu*Pz_impact/(0.5*tau);
    
    F_peak_total = v.Lx*trapz(z,pz_peak,2);
    q_peak_total = F_peak_total/(v.Lx*v.Lz);

    % Create peak time signal
    pz_impact = zeros(length(t),length(z));
    t_pulse = 0:dt:tau;
    pulse = triangularPulse(0,0.5*tau,tau,t_pulse);
    pulse_tot = zeros(1,length(t));
    pulse_tot(1:length(t_pulse)) = pulse;
    pz_impact = repmat(pz_peak,length(t),1).*(repmat(pulse_tot,length(pz_peak),1).');  
   
    %% Process force signal to obtain input for model
    
    % Load external force signal
%     f = load('./Data/pressures_DH_s110_compl_long_units.mat'); % Data: p (pressures), z (location of measurements p), t (time vector)
%     f = data_changetimestep(f,0.001); % Sets desired timestep for time-domain analysis
%     f = data_2Dv_addriseandfall(f,0.5,0.5); % adding linear rise and fall to pressure signals to start at zero
%     f = data_2Dv_interpolate(v,cc,f,struct('bctop','closed')); % interpolate date to grid used for model calculations
   
%     % Option 1: Skewness factor (0 - 1) is applied in function data_3D_makeintervals_skewness
%     f.skewness_width = 0; % Skewness factor 
%     f.p = pz_impact;
%     f.z = z; f.t = t;  
%     f.iv_x = [0:3:v.Lx]; % x-coordinates on borders intervals in which the force is divided
%     f.iv_z = [0:0.5:v.Lz]; % z-coordinates on borders intervals in which the force is divided
%     f = data_3D_makeintervals_skewness(v,cc,f); % divide force into intervals with their own amplitude-time signals
    
    % Option 2: Define any force shape over both directions in function data_3D_makeintervals
    f.p = zeros(length(t),length(v.x),length(z));
    f.p(:,1:round(length(v.x)),:) = permute(repmat(pz_impact,1,1,round(length(v.x))),[1 3 2]); % Example where the impact force is applied at only the left half of the gate
    f.z = z; f.x = v.x; f.t = t; 
    f.iv_x = [0 v.Lx]; % x-coordinates on borders intervals in which the force is divided
    f.iv_z = [0:0.5:v.Lz]; % z-coordinates on borders intervals in which the force is divided
    f = data_3D_makeintervals(v,cc,f); % divide force into intervals with their own amplitude-time signals
     
%% Obtain solution
    %% Output settings
    a.forceintervals = 1; % The time-domain signal is divided over multiple intervals
    a.fft = 1; % Fourier-analysis, save solution for modal coefficients each frequency step
    a.poi = [v.Lx/2 v.Lz; v.Lx/3 v.Lz; v.Lx/2 v.Lz*2/3]; % Points of interest [x z] on plate, where plate amplitudes and fluid pressures are evaluated (per frequency)
    a.outputpar = {'omegan','Wxz'}; % Extra variable names to save can be added 'x','Wkm','max_pfor_amp'
    
    %% Force and frequency (Fourier transform)
    maxomeg = 200*2*pi; % Limits the maximum frequency for which the response is taken into account 
    f.addstart = 1; % Adds time before signal (to see if response is zero before force 'starts'
    f.addend = 20; % Adds time after signal (to let the system damp out)
    f = model_fouriertransform(f,maxomeg);

    %% Run calculation of system response
    [out,v,a,f] = model_main(v,cc,a,f,modes); % RAO of the system in terms of modal coefficients
    Akm_H = out.Akm_H; clear out.Akm_H;

    %% Inverse Fourier transform
    fprintf('Performing inverse Fourier transform\n')

    % Time domain ifft
    n = length(f.interval(1).Y);
    if maxomeg > 0
        Akm_H(:,f.n_shortened+1:f.n_restore) = 0;
    end

    Akm_t = zeros(cc.smodes,n);
    for km = 1:cc.smodes
        Hifft(1,:) = Akm_H(km,:);
        Yresp = Hifft;
        Yresptr(1:n/2+1) = Yresp;
        Yfliptr = fliplr(conj(Yresp)); Yresptr(n/2+2:n) = Yfliptr(2:end-1);
        Akm_t(km,:) = ifft(Yresptr);
    end

%% Postprocessing
    %% Reduce result to relevant time interval for output
    % Matrix for full after fourier transform time series is too large to
    % process (> 100 GB). Therefore, the relevant part of the interval is
    % extracted.
    f.t = f.t+f.tshift; % Set zero-point of time vector to original 
    t_min = -0.1; % Lower bound of extracted response interval
    t_max = 2.0; % Upper bound of extracted response interval
    [~,index_start] = find(f.t >= t_min,1,'first');
    [~,index_end] = find(f.t >= t_max,1,'first');

    ndt = 2;%round(dt_req/dt); % Reduce time step
    Akm_t = Akm_t(:,index_start:ndt:index_end);
    f.t = f.t(index_start:ndt:index_end);
    f.totalforce = f.totalforce(index_start:ndt:index_end);

    %% Get plate deflection and strains from modal coefficients (final solution)
    % The unit modal shapes are loaded forom the main gatefluid model. However,
    % a more close grid can also be used by evaluating the original BVP
    % solutions again.
    fprintf('Composing plate response from structural coefficients\n')

    ngridx = 2; % reduce gridsize in x-direction with a factor n
    ngridz = 2; % reduce gridsize in z-direction with a factor n
    cc.dx = ngridx*cc.dx; v.x = v.x(1:ngridx:end); % set new x-coordinates
    cc.dz = ngridz*cc.dz; v.z = v.z(1:ngridz:end); % set new z-coordinates
    modes.Wxz = modes.Wxz(1:ngridx:end,1:ngridz:end,:);
%     modes.Sep = modes.Sep(1:ngridx:end,1:ngridz:end,:);
%     modes.Sen = modes.Sen(1:ngridx:end,1:ngridz:end,:);
%     modes.Tmax = modes.Tmax(1:ngridx:end,1:ngridz:end,:);
    
    Wxz_modes = zeros(size(modes.Wxz,1),size(modes.Wxz,2),size(Akm_t,2),cc.smodes);
%     Sep_modes = zeros(size(modes.Sep,1),size(modes.Sep,2),size(Akm_t,2),cc.smodes);
%     Sen_modes = zeros(size(modes.Sen,1),size(modes.Sen,2),size(Akm_t,2),cc.smodes);
%     Tmax_modes = zeros(size(modes.Tmax,1),size(modes.Tmax,2),size(Akm_t,2),cc.smodes);  
    col_disp_modes = zeros(size(modes.col_disp,1),size(Akm_t,2),cc.smodes);
    col_stress_modes = zeros(size(modes.col_stress,1),size(Akm_t,2),cc.smodes);
    for i = 1:size(Akm_t,2)
        for km = 1:cc.smodes
            Wxz_modes(:,:,i,km) = Akm_t(km,i)*modes.Wxz(:,:,km);
%             Sep_modes(:,:,i,km) = Akm_t(km,i)*modes.Sep(:,:,km);
%             Sen_modes(:,:,i,km) = Akm_t(km,i)*modes.Sen(:,:,km);
%             Tmax_modes(:,:,i,km) = Akm_t(km,i)*modes.Tmax(:,:,km);
            col_disp_modes(:,i,km) =  Akm_t(km,i)*modes.col_disp(:,km);
            col_stress_modes(:,i,km) =  Akm_t(km,i)*modes.col_stress(:,km);            
        end
    end
    wt = sum(Wxz_modes,4); % Full solution displacements over time
%     Sept = sum(Sep_modes,4); % Full solution Von Mises stresses+
%     Sent = sum(Sen_modes,4); % Full solution Von Mises stresses-
%     Tmaxt = sum(Tmax_modes,4); % Full solution Shear stresses
    col_dispt = sum(col_disp_modes,3); % Full solution col displacements over time
    col_stresst = sum(col_stress_modes,3); % Full solution col stresses over time


   %% Location and time of maximum deflection and stress
    % Find location and time where maximum deflection occurs
    % Maximum absolute deflection per time step
    [max_wt(1,:)] = max(max(wt,[],1),[],2); max_wt_abs = abs(max_wt);
    [iit,it] = find(max_wt_abs == max(max_wt_abs));
    Akm_at_w_max = Akm_t(:,it);
    w_at_tmax(:,:) = wt(:,:,it);
    col_disp_at_tmax = col_dispt(:,it);
    max_w_at_t_max = max(max(w_at_tmax)); max_deflection = abs(max_w_at_t_max);
    [ix,iz] = find(w_at_tmax == max_w_at_t_max,1,'first');

    xmax = v.x(ix);
    zmax = v.z(iz);
    tmax = f.t(it);
    wt_atmax = wt(ix,iz,:);
    
    % Find location and time of maximum stress
    [max_stresst(1,:)] = max(col_stresst,[],1); max_stresst_abs = abs(max_stresst);
    [~,it] = find(max_stresst_abs == max(max_stresst_abs));
    Akm_at_stress_max = Akm_t(:,it);
    col_stress_at_tmax = col_stresst(:,it);

    %% Plate displacement over time
    %movie_t_pshape_v2(wt,f,v,cc,a)
    pmotionoptions = struct('cutoffcriterion','time','cutoffvalue',2,'plottype','withforce','specificfolder','dyn_t_resp','dataname',['plateresponse_tau_' num2str(tau*1000) 'ms'],'numaxis',1);
    plot_t_pmotion(wt,f,v,cc,a,pmotionoptions);
    pmotionoptions.dataname = ['plateresponse_contributions_tau_' num2str(tau*1000) 'ms'];
    plot_t_pcontributions(Akm_t,f,v,cc,a,pmotionoptions);
    
    %% Plate shape at max deflection (2D)
    a.pshape = struct('plot',1,'dataname',['Platedefl_max_tau_' num2str(tau*1000) 'ms'],'specificfolder','dyn_t_resp','title',1,'titlename','Deflection');
    plot_pshape(v,a,w_at_tmax,a.pshape)
    %movie_t_pshape_v2(wt,f,v,cc,a);
    
    %% Plate shape at max deflection (3D)
    a.pshape = struct('plot',1,'dataname',['Platedefl_3D_max_tau_' num2str(tau*1000) 'ms'],'specificfolder','dyn_t_resp','ylabelname','Deflection [mm]','title',1,'titlename','Deflection');
    plot_pshape3D(a,modes,col_disp_at_tmax*1000,a.pshape)
    
    %% Plate stress over time
    % To be worked out: find the indices of the POI from modes.vert and plot stress over time
%     smotionoptions = struct('cutoffcriterion','time','cutoffvalue',2,'plottype','withforce','specificfolder','dyn_t_resp','dataname',['plate_strain_xx_tau_' num2str(tau*1000) 'ms'],'numaxis',1,'title',1,'titlename','Strain (xx)');
%     plot_t_smotion(col_stress_t,f,v,cc,a,smotionoptions);

    %% Stress shape at max stress
    if a.stress == 1
    a.pstress = struct('plot',1,'axis_normalised',0,'dataname',['Plate_stress_3D_max_tau_' num2str(tau*1000) 'ms'],'specificfolder','dyn_t_resp','ylabelname','Stress [N/mm$^2$]','title',1,'titlename','Stress at maximum','textfontsize',20);                    
    plot_pstress3D(a,modes,col_stress_at_tmax/10^6,a.pstress)
    end 
    
    %% Distribution of modes deflections
    a.pshape = struct('plot',1,'dataname',['Platedefl_wave_atwmax_tau_' num2str(tau*1000) 'ms_relative'],'relative','no','specificfolder','dyn_t_resp','title',1,'titlename','Deflection - modal contributions at max');
    plot_pshape_contr(a,Akm_at_w_max.',a.pshape); % Distribution at wmax
    a.pshape = struct('plot',1,'dataname',['Platedefl_wave_maxovertime_tau_' num2str(tau*1000) 'ms_relative'],'relative','no','specificfolder','dyn_t_resp','title',1,'titlename','Deflection - max per mode');
    max_Akm = max(Akm_t,[],2);
    plot_pshape_contr(a,max_Akm.',a.pshape); % Distribution at wmax
    
    %% Distribution of modes stress
    % Y-label and units not correct yet - make new function for stresses
    a.pshape = struct('plot',1,'dataname',['Stress_contr_wave_atwmax_tau_' num2str(tau*1000) 'ms_relative'],'relative','no','specificfolder','dyn_t_resp','title',1,'titlename','Stress - modal contributions at max');
    plot_pshape_contr(a,Akm_at_stress_max.',a.pshape); % Distribution at wmax    
    
end