function saveDeformation(H_ratio,P_mu,Q_mu,desErr,varargin)
%SAVEDEFORMATION saves the deformation of a point on a beam with
%width H at coordinates (xbeam,ybeam) wrt the coordinate frame of
%the beam and returns the deformation x and y direction. Qmu and Pmu
%are the shear stress and tensile stress (or if negative, pressure) 
%respectively.

replace = false;
progress = false;
message = false;
i = 0;
while i < length(varargin)
    i = i + 1;
    in = varargin{i};
    if strcmp(in,'replace')
        %option to only return maximum y position (for use in
        %optimization function)
        replace = true;
    elseif strcmp(in,'progress')
        progress = true;
    elseif strcmp(in,'message')
        message = true;
    else
        warning(['Unknown option ', in,' is ignored'])
    end
end  

%% print a message of what will be calculated and saved
if message
    fprintf('I will calculate and save %.f new deformations ',length(H_ratio)*length(P_mu)*length(Q_mu))
    fprintf('With the following parameters:\n DesignError = %.f %%, ',desErr)
    if length(H_ratio) == 1
        fprintf('H = %.4f. ',H_ratio)
    else
        fprintf('A range of %.f H-values ([%.4f, %.4f]). ',length(H_ratio),H_ratio(1),H_ratio(end))
    end
    if length(P_mu) == 1
        fprintf('P = %.4e. ',P_mu)
    else
        fprintf('A range of %.f P-values ([%.4e, %.4e]). ',length(Q_mu),P_mu(1),P_mu(end))
    end
    if length(Q_mu) == 1
        fprintf('Q = %.4e. ',Q_mu)
    else
        fprintf('A range of %.f Q-values ([%.4e, %.4e]). ',length(Q_mu),P_mu(1),Q_mu(end))
    end
end
    

%Load data from previous calculations
try
    load('DeformationData','Hlist','Plist','Qlist','udata','vdata',...
    'thetadata','lambdadata','alphadata','kdata','designErr'); 
catch
end
%#ok<*NASGU> because we load and save variables we can ignore some 
%#ok<*NODEF> matlab warnings.

%% check if we can use previously saved data

xlist = linspace(0,1,50);

%% find indexes to store the new dataa

n_done = 0;

for H = reshape(H_ratio,1,[])
    for Pmu = reshape(P_mu,1,[])
        for Qmu = reshape(Q_mu,1,[])
n_done = n_done + 1;
if progress 
    fprintf('Now calculating and saving value %.f of %.f\n',n_done,length(H_ratio)*length(P_mu)*length(Q_mu))
end
%check if there is already some data stored
if ~exist('udata','var') || isempty(udata)
    dSize = zeros(1,5);
    new = true;
else
    new = false;
    dSize = size(udata);
end



%H index
if new
    Hind = 1;
%check if we already have data for this H-value
elseif ~ismember(H,Hlist(designErr==desErr)) 
    Hind = dSize(1)+1;
    if Hind > size(Hlist,1)
        new = true;
        %resize the data containers
        Plist(Hind,:) = NaN(1,dSize(2));
        Qlist(Hind,:,:)         = NaN(1,dSize(2),dSize(3));
        udata(Hind,:,:,:)       = NaN(1,dSize(2),dSize(3),dSize(4));
        vdata(Hind,:,:,:)       = NaN(1,dSize(2),dSize(3),dSize(4)); 
        thetadata(Hind,:,:,:)   = NaN(1,dSize(2),dSize(3),dSize(4)); 
        lambdadata(Hind,:,:,:)  = NaN(1,dSize(2),dSize(3),dSize(4)); 
        alphadata(Hind,:,:,:)   = NaN(1,dSize(2),dSize(3),dSize(4));
        kdata(Hind,:,:,:)       = NaN(1,dSize(2),dSize(3),dSize(4));
    end
else
    Hind = find(H==Hlist(designErr==desErr),1);
    %adjust index for the applied filtering
    indList = find(designErr==desErr);
    Hind = indList(Hind);
end     
Hlist(Hind) = H;
designErr(Hind) = desErr;

%P index
if new
    Pind = 1;
%check if we already have data for this H-P combination
elseif ~ismember(Pmu,Plist(Hind,:))
    Pind = sum(~isnan(Plist(Hind,:)))+1; %find index of first NaN
    if Pind > dSize(2)
        new = true;
        Plist(:,Pind)           = NaN(dSize(1),1);
        Qlist(:,Pind,:)         = NaN(dSize(1),1,dSize(3));
        udata(:,Pind,:,:)       = NaN(dSize(1),1,dSize(3),dSize(4));
        vdata(:,Pind,:,:)       = NaN(dSize(1),1,dSize(3),dSize(4)); 
        thetadata(:,Pind,:,:)   = NaN(dSize(1),1,dSize(3),dSize(4)); 
        lambdadata(:,Pind,:,:)  = NaN(dSize(1),1,dSize(3),dSize(4)); 
        alphadata(:,Pind,:,:)   = NaN(dSize(1),1,dSize(3),dSize(4));
        kdata(:,Pind,:,:)       = NaN(dSize(1),1,dSize(3),dSize(4));        
    end

else
    Pind = find(Plist(Hind,:)==Pmu,1);
end
Plist(Hind,Pind) = Pmu; 

% Q index
if new
    Qind = 1;
%check if we already have data for this H-P-Q combination
elseif ~ismember(Qmu,Qlist(Hind,Pind,:))
    Qind = sum(~isnan(Qlist(Hind,Pind,:)))+1; %find index of first NaN
    if Qind > dSize(3)
        new = true;
        Qlist(:,:,Qind)         = NaN(dSize(1),dSize(2),1);
        udata(:,:,Qind,:)       = NaN(dSize(1),dSize(2),1,dSize(4));
        vdata(:,:,Qind,:)       = NaN(dSize(1),dSize(2),1,dSize(4)); 
        thetadata(:,:,Qind,:)   = NaN(dSize(1),dSize(2),1,dSize(4)); 
        lambdadata(:,:,Qind,:)  = NaN(dSize(1),dSize(2),1,dSize(4)); 
        alphadata(:,:,Qind,:)   = NaN(dSize(1),dSize(2),1,dSize(4));
        kdata(:,:,Qind,:)       = NaN(dSize(1),dSize(2),1,dSize(4));
    end            
else
    %we already have data for this Q index
    Qind = find(Qlist(Hind,Pind,:)==Qmu,1);
    if ~replace
        warning('This situation is already calculated. Use ''replace'' to overwrite')
        continue
    end
end
Qlist(Hind,Pind,Qind) = Qmu;

%% calculate the deformation
if n_done == 1
    [x,y,~,alpha,lambda,theta,k] = neoHookean(1,H,Qmu,Pmu);
else
    [x,y,~,alpha,lambda,theta,k] = neoHookean(1,H,Qmu,Pmu,...
        'guessAlphaLambda',alpha,lambda);
end

%% save new data
udata(Hind,Pind,Qind,:)     = reshape( x-xlist,  1,1,1,[]);
vdata(Hind,Pind,Qind,:)     = reshape( y,        1,1,1,[]); 
thetadata(Hind,Pind,Qind,:) = reshape( theta,    1,1,1,[]); 
lambdadata(Hind,Pind,Qind,:)= reshape( lambda,   1,1,1,[]); 
alphadata(Hind,Pind,Qind,:) = reshape( alpha,    1,1,1,[]); 
kdata(Hind,Pind,Qind,:)     = reshape( k,        1,1,1,[]); 


        end
    end
end

save('DeformationData','Hlist','Plist','Qlist','udata','vdata',...
            'thetadata','lambdadata','alphadata','kdata','designErr')    
    
    
end