function [list,dispx, dispy] = mechanicalBeamFem(L,H,Q,P,sweepparam,sweepList,varargin)
%MECHANICALBEAMFEM uses Comsol Multiphysics to find the deformation of
%a beam. In contrast to the electrostatic model, this mechanical model
%is not built from scratch in Matlab. It opens the model files
%'beamMatlab.mph' and/or 'beamMatlab3.mph' and adjusts the dimensions
%of the beam to do the simulations. For this reason, it is best not to
%edit these mph files. 
%
%Inputs:
% - L: length of the beam [m]
% - H: width of the beam [m]
% - Q: Shear stress in Pa
% - P: Normal stress in Pa
% - seepparam: A parameter for a sweep ('Q', 'H', 'L' or 'P')
% - sweeplist: values for the sweep parameter

%Optional inputs: 
% - 'checkMesh': Checks the influence of the mesh by doing a
%                   refinement.
% - '3D' : Do a simulation in three dimensions
% - 'WS-ratio': Set the width of the beam (required for 3D simulation)
%               Input as W/L ratio (so divide by the beam length).
%
%Outputs:
% - List: The list of values from the sweepparam used
% - dispx and dispy: The displacement of the end of the beam. 

    checkMesh = false;
    threeD = false;
    %readout options
    i = 0;
    W = 2;
    while i < length(varargin)
        i = i + 1;
        in = varargin{i};
        if strcmp(in,'checkMesh')
            checkMesh = true;
        elseif strcmp(in,'3D')
            threeD = true;            
        elseif strcmp(in,'WS-ratio')
            i = i + 1;
            W = varargin{i};
        else
            warning(['Unknown option ', in,' is ignored'])
        end
    end     
    
    if threeD
        model = mphopen('beamMatlab3.mph');
        model.param.set('W', ['L*',num2str(W,'%.16f')]);
    else
        model = mphopen('beamMatlab.mph');
    end
    model.component('comp1').mesh('mesh1').autoMeshSize(3);
    
    %substitute the the input parameters
    model.param.set('L', num2str(L,'%.16f [m]'));
    model.param.set('Q', num2str(Q,'%.16f [Pa]'));
    model.param.set('P', num2str(P,'%.16f [Pa]'));
    model.param.set('H', num2str(H,'%.16f [m]'));
    
    %adjust the sweep parameter and sweep list (do sweeps in comsol)    
    liststr = num2str(sweepList,'%.16f,');
    model.batch('p1').set('pname', {sweepparam});
    model.batch('p1').set('plistarr', {liststr(1:end-1)});
    
    %run the simulation
    model.batch('p1').run;
    
    %put results in a table 
%     model.result('pg5').run;
    model.result.table('tbl1').clearTableData;
    model.result.numerical('pev1').set('table', 'tbl1');
    model.result.numerical('pev1').setResult;  
    
    %get the results from the table
    tab = mphtable(model,'tbl1').data;
    dispx = tab(:,2);
    dispy = tab(:,3);
    list = tab(:,1);
    
    if checkMesh
        %decrease meshsize
        model.component('comp1').mesh('mesh1').autoMeshSize(1);
        
        %run the simulation
        model.batch('p1').run;
        
        %put results in a table 
%         model.result('pg5').run;
        model.result.table('tbl1').clearTableData;
        model.result.numerical('pev1').set('table', 'tbl1');
        model.result.numerical('pev1').setResult;  

        %get the results from the table
        tab = mphtable(model,'tbl1').data;
        dispx2 = tab(:,2);
        dispy2 = tab(:,3);
        list2 = tab(:,1);
        fprintf('The (max) displacement with a normal mesh is %.5e m, with a finer mesh %.5e m (%.4f%% difference) \n', ...
        max(dispx,[],'all'), max(dispx2,[],'all'),(max(dispx,[],...
        'all')-max(dispx2,[],'all'))/max(dispx,[],'all')*100)
    
    end

end

