function [dCdx,Cbase,y0,linRange] = ...
    capacitanceLinearization(yposD,xpos,U,d,varargin)
%CAPACITANCELIENARIZATION Analyzes the effect of translation (in x) 
% direction on the capacitance of a wire structure. And find a 
% linearization around the point with the largest slope.
%
% Required inputs:
% - yposD defines the (default) position of the wires
% - xpos defines the x position of the wires.
% - U defines the potential of the wires, the wires with positive 
%       potential are displaced
%
% Optional name-value pairs:
% - 'plot'      create a plot of the wire structure
% - 'linRange'  Find the linarization range too (otherwise linRange
%               will be set to NaN (to save computation time)
% - 'multiElectrode' Adds an additional electrode with positive
%               potential and displaces it to -y0. The inputs should
%               still only contain 2 electrodes, the third is added in
%               this function.

%% check the input

%default variables
doplot = false;
multiElectrode = false;
linRange = NaN;

% analyze optional input arguments;
i = 0;  

while i < length(varargin)
    i = i + 1;
    in = varargin{i};
    if strcmp(in,'plot')
        doplot = true;
    elseif strcmp(in,'linRange')
        linRange = [0,0];
    elseif strcmp(in,'multiElectrode')
        if i < length(varargin) && islogical(varargin{i+1})
            i = i+1;
            multiElectrode = varargin{i};
        else
            multiElectrode = true;   
        end   
    else
        warning(['Unknown option ', in,' is ignored'])
    end
end  


%% Definition of the wire structure
electrode = U>0;

if multiElectrode
    %add the third elecrode
    yposD = [yposD,-yposD(U>0)];
    xpos = [xpos,xpos(U>0)];
    electrode = [U>0,2*ones(1,sum(U>0))];
    U = [U,U(U>0)];
end


count = 0;

delta = 1e-10*d; %delta x used to approximate the derivatives

%use optimization to find the maximum derivitive (by multiplying by -1)

%for a single electrode, the optimization domain is unbounded, so we
%use fminsearch():
if ~multiElectrode
    x0_list = [-0.1,-2,-3.8,-5]; %try different starting points to try to 
                                 %find the absolute minimum
    dCdx = Inf;
    for x0  = x0_list
        [xbase1,dCdx1] = fminsearch( @(x) displaceChange(x)*-1, x0);
        if dCdx1<dCdx
            dCdx = dCdx1;
            y0 = xbase1;
        end
    end
else
    %For the multielectrode case, we have a bounded optimization
    %problem: The two electrodes should not overlap.
    sortY = (sort(yposD(electrode==1)));
    if length(unique(diff(sortY))) > 1
        Welec = sortY(find(abs(diff(sortY)-max(diff(sortY)))<1e-10,1))-sortY(1) + 2*d;
        Wgap = sortY(find(abs(diff(sortY)-max(diff(sortY)))<1e-10,1)+1) - ...
                        sortY(find(abs(diff(sortY)-max(diff(sortY)))<1e-10,1)) - 2*d;
    else
        Welec = sortY(end)-sortY(1) + 2*d;
        Wgap = 10*Welec;
    end
    
    if Wgap < Welec
        warning('The separation in between groups is too small to have an additional electrode')
        dCdx = NaN;
        Cbase = NaN;
        y0 = NaN;
        linRange = NaN;
        return
    end
    [y0,dCdx] = fminbnd( @(x) displaceChange(x)*-1 ,-Wgap/2,-Welec/2);
end


    dCdx = dCdx *-1; %Compensate for the multiplication by -1 above.
    Cbase = displaceStructure(y0,0);
   

%if we don't have to create a linarization we are done now
if isnan(linRange)
    return
end
maxError = 0.01;

%find the range where the error is still less than  the maxerror
options = optimoptions('fmincon','Display','none');
[linRange(1)] = fmincon(...
    @linearizationError,y0,[],[],[],[],-inf,y0,[],options);
[linRange(2)] = fmincon(...
    @linearizationError,y0,[],[],[],[],y0,inf,[],options);

%% make plot
if doplot
    plotWires(yposD+y0,xpos,d)
end

%% Functions for use in minimization
function C = displaceStructure(disp,dx)
    count = count + 1;
    %displace the structure with positive potential
    ypos = +yposD + disp .* (electrode==1) - disp .* (electrode==2)...
                                            - dx .* (electrode>0);
    [lambda] = potentialToCenteredCharge2(ypos,xpos,U,d);

    %calculate and store the capacitance and other variables
    if multiElectrode
        [C,~] = findCapacitance(U,lambda,'multiElectrode',electrode);
    else
        [C,~] = findCapacitance(U,lambda,false);
    end
end

function dCdX = displaceChange(disp)
    %find the rate of capacitance change per displacment
    dCdX = (displaceStructure(disp,0) - ...
                    displaceStructure(disp,delta))/delta;        
end

function result = linearizationError(disp,right) 
    %find the error between the linearization and the c-w appox.
    creal = displaceStructure(disp,0);
    Clin = (disp-y0)*dCdx + Cbase;	
    err = (creal-Clin)./creal;
    result = maxError - abs(err); %find the point nearest to 0
    
    %trick the minimize function to find the bounds
    if result<0
        result = 10+abs(disp-y0);
    end
end



end

