function [CA,Cbase,dCdx,linRange,xbase] = capacitanceDisplacment(yposD,xposD,U,d,dispY,varargin)
%CAPACITANCEDISPLACMENT Analyzes the effect of translation (in y) direction
% on the capacitance of a wire structure. This is done by manipulating the
% xpos vector and calling other functions. The second part of this function
% visualizes the results in an optional animation, which is saved in an
% .avi file.
%
% Required inputs:
% - xposD defines the (default) position of the wires
% - ypos defines the y position of the wires.
% - U defines the potential of the wires, the wires with positive potential
%       are displaced
% - dispA is an array which contains the values with which to displace the
%       wires with positive potential towards positive x
%
% Optional name-value pairs:
% - 'plot'      create a plot of the capacitance-displacement curve
% - 'animation' (boolean, default false) defines whether an animation is
%       created (note, this substantially increases execution time)
% - 'potField' and 'integrationPaths' (boolean, default false) indicate
%       wheater the equipotentiallines and integratoin paths are shown in 
%       the animation.
% - 'paperLayout' Alternative layout for the animation, which is used
% for the video in the paper.
% - 'videoName' (string, default 'vid') is the filename to save the video
% - 'framerate' (positive scalar, default 10) sets the fps of the video.
% - 'modelMechanics' Requires extra inputs: H, P, Qlist, Sbeam, desErr. dispX input will be
% ignored, instead all points in the Qlist will be used. For dC/dx,
% delta C / delta x will be outputted, which is the average slope over
% the entire Q-range. Additionally 'inBlock' or 'inBeam' should be
% supplied, which matches the design type.
% - 'DivideQ' Divide the shear stress on the axis by the number
% following this argument. This is usefull if you are interested in
% plotting the shear stress on a sensor instead of on a beam. Then
% divide by the factor n.
% - 'multiElectrode' Requires one additional input containing an array
% with the same length as U with the structure number to which the
% wire belogs. 0 indicates the ground structure to which capacitance
% is calculated. For use in this funtion, the output will contain the
% capacitance difference between electrode 1 and 2 (to the ground).
% - 'FEMonly'  do a FEM simulation
% - 'rectangle' use rounded plates instead of wires (in FEM)
% - 'length' do a finite length FEM simiulation
% - 'dCdy' plot the derivative of the capacitiance wrt displacement of
%         top structure
% - 'y0'   Varies the initial displacement instead of normal
%          displacement. This only influences results if used in
%          combination with multielectrode, in that case electrodes
%          are moved in opposite directions if y0 is turned on. Use no
%          initial displacment.
% - 'optimizey0' Uses capaciatanceLinearization first to determine the
%          optimal initial displacement. Do not use in combination
%          with 'y0', as this introduces an initial displacment.

    
%% check the input
%default variables
animation = false;
doplot = false;
potField = false;
videoname = 'vid';
framerate = 10;
axSpecified = false;
dCdy = false;
FEMonly = false;
dispFEM = dispY;
L = inf;
modelMechanics = false;
inBeam = false;
paperLayout = false;
multiElectrode = false;
rectangle = false;
dispX = zeros(size(dispY));
y0 = +1;
optimizey0 = false;
divideQ = 1;

linRange = [NaN,NaN];
Cbase = NaN;
xbase = NaN;

% analyze optional input arguments;
i = 0;  
%function to check if x is a positive scalar
validScalarPosNum = @(x) isnumeric(x) && isscalar(x) && (x > 0);

while i < length(varargin)
    i = i + 1;
    in = varargin{i};
    if strcmp(in,'plot')
        doplot = true;
    elseif strcmp(in,'animation')
        animation = true;
    elseif strcmp(in,'potField')
        potField = true;
    elseif strcmp(in,'framerate')
        i = i + 1;
        if validScalarPosNum(varargin{i})
            framerate = varargin{i};
        else
            warning('Invalid input for framerate')
        end
    elseif strcmp(in,'videoName')
        i = i + 1;
        videoname = varargin{i};
    elseif strcmp(in,'axis')
        i = i + 1;
        axSpecified = true;
        ax = varargin{i};
    elseif strcmp(in,'FEMonly')
        FEMonly = true;
    elseif strcmp(in,'paperLayout')
        paperLayout = true;
    elseif strcmp(in,'Length')
        i = i + 1;
        L = varargin{i};
    elseif strcmp(in,'dispY')
        i = i + 1;
        dispX = varargin{i};
    elseif strcmp(in,'modelMechanics')
        modelMechanics = true;
        i = i + 1; 
        H = varargin{i};
        i = i + 1; 
        P = varargin{i};
        i = i + 1; 
        Qlist = varargin{i};
        i = i + 1; 
        S = varargin{i};
        i = i + 1; 
        desErr = varargin{i};
    elseif strcmp(in,'divideQ')
        if i < length(varargin)
            i = i+1;
            divideQ = varargin{i};
        end
    elseif strcmp(in,'inBlock')
        inBeam = true;
        xbeam = 1;
        ybeam = 0;
    elseif strcmp(in,'rectangle')
        rectangle = true;
    elseif strcmp(in,'multiElectrode')
        if i < length(varargin) && islogical(varargin{i+1})
            i = i+1;
            multiElectrode = varargin{i};
        else
            multiElectrode = true;   
        end
    elseif strcmp(in,'dCdy')
        dCdy = true;   
    elseif strcmp(in,'y0')
        y0 = -1;
    elseif strcmp(in,'optimizey0')
        optimizey0 = true;  
    else
        warning(['Unknown option ', in,' is ignored'])
    end
end  


%Investigate capacitance change between two rows of wires


%% optimization of initial displacement
if optimizey0
    [dCdx,Cbase,xbase,linRange] = capacitanceLinearization(yposD,xposD,U,d,'multiElectrode',multiElectrode);
    yposD = yposD + (U>0)*xbase;
end

%% Determine sweep range
if ~FEMonly && ~modelMechanics && ~multiElectrode
    [dCdx,Cbase,xbase,linRange] = ...
                capacitanceLinearization(yposD,xposD,U,d,'linRange');
    if isempty(dispY)
       dispY = linspace( min([linRange*2,0]), max([linRange*2,0]),51);    
    end
    if isempty(dispX)
       dispX = zeros(size(dispY));
    end
end
if multiElectrode && ~modelMechanics && y0 < 0
    %Find a suitable range for modelmechanics without overlap between
    %electrodes. 
    if isempty(dispY)
        if sort(yposD(U<0)) ~= sort(yposD(U>0))
            error('Empty dispY array for a multiElectrode structure with initial displacement. Please use no initial displacement or put your own  range in dispY')
            return
        end
        sortY = sort(yposD);
        if length(unique(diff(sortY))) > 2
            %find the size of the electrode and of the gap between the
            %electrodes (used to find the distance we can move in the
            %end)
            Welec = sortY(find(diff(sortY)==max(diff(sortY)),1))-sortY(1) + 2*d;
            Wgap = sortY(find(diff(sortY)==max(diff(sortY)),1)+1) - ...
                            sortY(find(diff(sortY)==max(diff(sortY)),1)) - 2*d;
        else
            Welec = sortY(end)-sortY(1) + 2*d;
            Wgap = 100*Welec; %practically infinite
        end
        dispY = -linspace(Welec/2,Wgap/2,51);
        dispX = zeros(size(dispY));
    end
end

if multiElectrode && ~modelMechanics && y0 > 0
    %Find a suitable range for modelmechanics without overlap between
    %electrodes. 
    if isempty(dispY)
        sortY = sort(yposD(U<0));
        if length(unique(diff(sortY))) > 2
            %find the size of the electrode and of the gap between the
            %electrodes (used to find the distance we can move in the
            %end)
            Welec = sortY(find(diff(sortY)==max(diff(sortY)),1))-sortY(1) + 2*d;
            Wgap = sortY(find(diff(sortY)==max(diff(sortY)),1)+1) - ...
                            sortY(find(diff(sortY)==max(diff(sortY)),1)) - 2*d;
        else
            Welec = sortY(end)-sortY(1) + 2*d;
            Wgap = 0;
        end
        dispY = linspace(0,2*Welec+Wgap,51);
        dispX = zeros(size(dispY));
    end
end


% %multielectrode case:
if multiElectrode
    yposD = [yposD,-yposD(U>0)];
    xposD = [xposD,xposD(U>0)];
    electrode = [U>0,2*ones(1,sum(U>0))];
    U = [U,U(U>0)];
    e = electrode;
else
    e = U>0;
end

nsteps = length(dispY); %number of situations we have to analyze

%% Find the displacement vectors u and v from mechanics
if modelMechanics
    mu = 1.32e8; %Initial shear modulus for Armadillo
    if isempty(P)
        P = 0;
    end
    if isempty(Qlist)
        Qlist = linspace(0,3.7*140e3,51);
    end
    if isempty(H)
        H = 2.746;
    end
    if isempty(S)
        S = 15;
    end
    if isempty(desErr)
        desErr = 20;
    end
    Hratio = H/S; 
    [u,v] = findDeformation(Hratio,P/mu,Qlist/mu,xbeam,ybeam,desErr);
    if inBeam
        dispY = reshape(v,1,[]) * S;
        dispX = reshape(u,1,[]) * S;
    end
    nsteps = length(Qlist);
end

%% initiate animation
if animation == true
    %create a large figure to make a sharper video
    fig = figure('units','pixels','position',[0 0 1920 1080]);
    myVideo = VideoWriter(videoname); %open video file
    myVideo.FrameRate = framerate;    %set the framerate
    open(myVideo)
end

CA = nan(1,nsteps);     %preallocation
errA = CA;
Q = CA;

nFEM = length(dispFEM);
CFEM = nan(1,nFEM);

%% Finding the capacitance for all structures (analytic)
if ~FEMonly
    maxLambda = 0;
    maxX = max(yposD) + max(dispY) + d;
    minX = min(yposD) + min(dispY) - d;
    for i = 1:nsteps
        %displace the electrodes
        ypos = yposD + dispY(i) * (e==1) + dispY(i) * (e==2) * y0;
        xpos = xposD + dispX(i) * (e>0);
        
        %find the charge on the wires
        [lambda] = potentialToCenteredCharge2(ypos,xpos,U,d);

        %check the potential error
        [~,relPotentialError] = ...
                           checkPotential(ypos,xpos,d,U,lambda,false);
        %calculate and store the capacitance and other variables        
        if multiElectrode && (dCdy || y0<0)
            %Find the slope of the capacitance displacemnet curve
            delta = 1e-10;
            [lambda2] = potentialToCenteredCharge2...
                                           (ypos+delta*(e>0),xpos,U,d);
            C = (findCapacitance(U,lambda2,false,...
                 'multiElectrode',electrode) - findCapacitance(...
                 U,lambda,false,'multiElectrode',electrode)) / delta;
            q1 = NaN;
        elseif dCdy
            %Find the slope of the capacitance displacemnet curve
            delta = 1e-10;
            [lambda2] = potentialToCenteredCharge2...
                                           (ypos+delta*(e>0),xpos,U,d);
            C = (findCapacitance(U,lambda2,false) - findCapacitance(...
                 U,lambda,false)) / delta;
            q1 = NaN;
        elseif multiElectrode
            C = findCapacitance(U,lambda,false,...
                'multiElectrode',electrode);
            q1 = NaN;
        else
            [C,q1] = findCapacitance(U,lambda,false);
        end
        CA(i) = C;
        errA(i) = relPotentialError;
        Q(i) = q1(1);

        %store extrema to have consistent limits in the animaton
        if max(abs(lambda)) > maxLambda
            maxLambda = max(abs(lambda));
        end
    end

end
%Array on the x axis for plotting
if modelMechanics
    xA = Qlist/divideQ;
else
    xA = dispY;
end

%% finding capacitance using FEM
if FEMonly
    for i = 1:length(dispY)
        %displace the structure(s) with positive potential
        ypos = yposD + dispY(i) * (e==1) + dispY(i) * (e==2) * y0;
        xpos = xposD + dispX(i) * (e>0);       
       
        
        %use FEM functions to call comsol to find C
        if dCdy
            %we need to find numerical derivative
            klist = [1,2];
            delta = 1e-10;
        else 
            klist = 1;
        end
        
        for k = klist
            if k > 1
                ypos = ypos+delta*(e>0);
            end
            if isinf(L)
                if rectangle
                    C(k) = diag(FEMwireStructure(ypos,xpos,U,e,d,'rectangle'));
                    FEMtag = 'FEMinfLength-Rect.';
                else
                    C(k) = diag(FEMwireStructure(ypos,xpos,U,e,d));
                    FEMtag = 'FEMinfLength';
                end                
            else
                if rectangle
                    C(k) = diag(FEMwireStructure3(ypos,xposD,U,e,d,L,...
                     'rectangle')/(L/1000));
                     FEMtag = ['FemL=',num2str(L),'-Rect.'];
                else
                    C(k)=diag(FEMwireStructure3(ypos,xposD,U,e,d,L)/(L/1e3));
                    FEMtag = ['FemL=',num2str(L)];
                end
                
            end
        end
        if dCdy
            CFEM(i) = (C(2)-C(1))/delta;
            xpos = xpos-delta;
        else
            CFEM(i) = C(1);    
        end    
    end
end

if modelMechanics
    %fill the unset outputs
    dCdx = (CA(2) - CA(1)) / (Qlist(2)  - Qlist(1));
end

%if we don't have to create a plot we are done now
if ~doplot
    if FEMonly
        CA = CFEM;
    end
    return
end

%if we make an animation, we use a subplot for the capacitance/disp
%figure, otherwise this is the only plot we make
if animation 
    if ~paperLayout
        capplot = subplot_tight(15,1,1:6,0.06);
    else
        capplot = subplot_tight(15,1,1:8,0.06);
    end
end
if ~axSpecified
    ax = gca;
    if paperLayout
        ax.FontSize = 25;
        ax.FontName = 'Times New Roman';
    end
    hold on
end

%plot analytical
if ~FEMonly
    plot(ax,xA,CA,'-','LineWidth',3,'Tag','Analytical')
    if ~modelMechanics && ~multiElectrode && ~dCdy
        %plot linarization
        if paperLayout
            linRangeWide = [mean(linRange) - 0.75*abs(diff(linRange)), mean(linRange) + 0.75*abs(diff(linRange))];
            plot(ax,linRangeWide,(linRangeWide-xbase)*dCdx + Cbase,'r-.','LineWidth',3,'Tag','Linearization')
        else
            plot(ax,linRange,(linRange-xbase)*dCdx + Cbase,'r-.','LineWidth',2,'Tag','Linearization')
        end 
    end
end

%plot fem results
if FEMonly    
    plot(ax,xA,CFEM,'*','Tag',FEMtag,'DisplayName',FEMtag)
end

if ~axSpecified
    hold off
    if modelMechanics
        xlabel('Shear stress [Pa]','FontSize',25)
    else
        xlabel('Displacement [mm]','FontSize',25)
    end
        
    ylabel('Capacitance [F/m]','FontSize',25)
    if (multiElectrode && y0<0) || dCdy 
        ylabel('dC/dy [F/m]','FontSize',25)
    end
%     title('Capacitance')
    grid on
    box on
end
    
%if we do not have to make an animation, we are done
if animation == false
    return
end

%In the top-plot we indicate at whcih displacement we are
hold(capplot,'on')
yl = ylim();
h1 = plot(zeros(2,1),yl);
% pos = [0.15,0.82,0.1,0.1];
pos = [0.07,0.83,0.1,0.1];
%create an annotation with the current displacement
an = annotation('textbox',pos,'string','y_0 = 0','FontSize',25,'FontName','Times New Roman');
hold(capplot,'off')
%define the levels for the contour function 
levels = (linspace(2*min(U),2*max(U),31));

%create a plot of the wire structure
if ~paperLayout
    wsplot = subplot_tight(15,1,8:10,0.06)
    ax = gca;
    ax.FontSize = 25;
    ax.FontName = 'Times New Roman';
    lambdaPlot = subplot_tight(15,1,12:15)
else
    wsplot = subplot_tight(15,1,10:14,0.06)
    ax = gca;
    ax.FontSize = 25;
    ax.FontName = 'Times New Roman';
end

for i = 1:length(dispY)
    ypos = yposD + dispY(i) * (e==1) + dispY(i) * (e==2) * y0;
    xpos = xposD + dispX(i) * (e>0);
    [lambda] = potentialToCenteredCharge2(ypos,xpos,U,d);
%     [~,relPotentialError] = checkPotential(ypos,xpos,d,U,lambda,false);
%     if ~multiElectrode
%         [C,~] = findCapacitance(U,lambda,false);
%     else
%         delta = 1e-10;
%         [lambda2] = potentialToCenteredCharge2...
%                                        (ypos+delta*(e>0),xpos,U,d);
%         %Find dC/dx (the differential capacitance is boring because it
%         %is always 0 in the state we look at.
%         C = (findCapacitance(U,lambda2,false,...
%              'multiElectrode',electrode) - findCapacitance(...
%              U,lambda,false,'multiElectrode',electrode)) / delta;
%     end
%     CA(i) = C;
%     errA(i) = relPotentialError;
    
    %update the position indicator and annotation
    delete(h1);
    delete(an);
    %create new position indiactor and annotation
    hold(capplot,'on')
    h1 = plot(capplot,xA(i)*ones(2,1),yl,'r--','LineWidth',2);
    an = annotation('textbox',pos,'string',['y_0 = ',num2str(xA(i),'%1.3f')],'FontSize',25,'FontName','Times New Roman');
    hold(capplot,'off')


    %clear the axis
    cla(wsplot)
    
    if i == 1
        hold(wsplot,'on')
        plotWires(ypos,xpos,d,wsplot,e)
        xlabel(wsplot,'$y$ [mm]','FontSize',25,'Interpreter','latex')
        ylabel(wsplot,'$x$ [mm]','FontSize',25,'Interpreter','latex')
        box on 
        drawnow
        xlim([-max(-minX,maxX),max(-minX,maxX)]);        
        axis(wsplot,'equal')
        drawnow
        potlimx = xlim(wsplot);
        potlimy = ylim(wsplot);       
        cla(wsplot)
    end
   
    %plot optional potential field:
    if potField
        %plot the level curves of the potential field
        [X,Y,phi] = chargeToPotential(ypos,xpos,lambda,potlimx(1),potlimx(2),potlimy(1),potlimy(2)); 
        hold(wsplot,'on')
        contour(wsplot,X,Y,phi,levels,'LineWidth',2)
        hold(wsplot,'off')
    end
    
    %show the wire structure
    hold(wsplot,'on')
    plotWires(ypos,xpos,d,wsplot,e)
    hold(wsplot,'off')

    xlim(wsplot,potlimx)
    ylim(wsplot,potlimy)
    
    if ~paperLayout
        %in the last plot we plot the charges on the different wires
        


        plot(lambdaPlot,ypos,lambda,'rx','MarkerSize',10)    
        xlim(lambdaPlot,[minX,maxX]);
        grid(lambdaPlot,'minor')
        ylabel(lambdaPlot,'Line charge \lambda [C/m]','FontSize',15)
        xlabel(lambdaPlot,'Position [mm]','FontSize',15)
        ylim(lambdaPlot,[-maxLambda,+maxLambda])
    end
    drawnow %update the figure
    
    F(i) = getframe(fig);
    writeVideo(myVideo, F(i));
end
% movie(F,1)
%save and close the video file
close(myVideo)


end

