function C = FEMwireStructure(y,x,U,electrode,d,varargin)
%FEMwireStructure finds the capacitance between wire structures using 
%comsol to do finite element analysis. This script requires a livelink
%with comsol. This script uses a 2D-geometry which in essence is 
%assuming that the wires are infinetely long. 
%
%The position of the wires is determined by the x and y inputs.
%U sets the potential of the wires. The potential differences are 
%conserved, but the structure with the lowest potential is set as 
%ground structure to determine the capacitance to. 
%The d input specifies the diameter of the wires (equal for all wires)
%
%Optional arguments
%'checkMesh' - checks the influence of a refined mesh
%'checkSpace' - checks the influence of a larger air space around the
%structure. 
%'saveFile' - saves the file as 'output.mph' in the current folder.
%'plot' - makes a plot of the geometry
%%'rectangle' - makes a connection between wires in the same group.
%Groups need to be oriented along a parallel of the x-axis and the
%xseparation in groups should be the minimum x separation in the
%structure.

% Parts of this script generated by COMSOL 5.4.0.388.
% Used with comsol 5.4.0.388 and corresponding comsol livelink server.

if nargin < 4
    error('Not enough input arguments (x,y,U,d) are required')
end

%initialize options to false
plot = false; 
checkMesh = false; 
checkSpace = false; 
saveFile = false;
rectangle = false;

%readout options
for i = 1:length(varargin)
    in = varargin{i};
    if strcmp(in,'plot')
        plot = true;
    elseif strcmp(in,'checkMesh')
        checkMesh = true;
    elseif strcmp(in,'checkSpace')
        checkSpace = true;
    elseif strcmp(in,'saveFile')
        saveFile = true;
    elseif strcmp(in,'rectangle')
        rectangle = true;
    else
        warning(['Unknown option ', in,' is ignored'])
    end
end    

%sort input on their ypos
[y,order] = sort(y);
x = x(order);
U = U(order);
electrode = electrode(order);


%% Selet a ground structure (to with every potential is calcluated)
if sum(U==0) == 0 
    %there is no ground structure defined: set the structure with 
    %lowest potential to ground and keep voltage differences:
    if all(U == electrode)
        U = U - min(U);
        electrode = U;
    else
        U = U - min(U);
    end
end

%% Determine the simulation boundaries
space = 10*(max(max(y)-min(y),max(x)-min(x))+d);
width = max(y)-min(y)+space;
height= max(x)-min(x)+space;

%% create comsol model and geometry

%initialize comsol model
import com.comsol.model.*
import com.comsol.model.util.*

model = ModelUtil.create('Model');

%Create a model with a two dimensional geometry (infinitely long) and 
%add electrostatic physics
model.modelPath('C:\Users\jenso\Documents\Advanced technology\Bachelor Assignment\FEM');
model.component.create('comp1', true);
model.component('comp1').geom.create('geom1', 2);
model.component('comp1').mesh.create('mesh1');
% model.component('comp1').mesh('mesh1').autoMeshSize(2);
model.component('comp1').physics.create('es', 'Electrostatics', 'geom1');

%Time independent study (we are only interested in the capacitance)
model.study.create('std1');
model.study('std1').setGenConv(true);
model.study('std1').create('stssw', 'StationarySourceSweep');
model.study('std1').feature('stssw').set('solnum', 'auto');
model.study('std1').feature('stssw').set('notsolnum', 'auto');
model.study('std1').feature('stssw').activate('es', true);

%set the diameter
model.param.set('d', [num2str(d),'[m]']);

%create free space rectangle
model.component('comp1').geom('geom1').create('space', 'Rectangle');
rect = model.component('comp1').geom('geom1').feature('space');
rect.set('type', 'solid');
rect.set('base', 'center');
rect.set('pos', [mean(y),mean(x)]);
rect.set('size', [width,height]);
rect.set('selresult',true);

%create a dummy circle to copy
model.component('comp1').geom('geom1').create('dummyCircle', 'Circle');
circle = model.component('comp1').geom('geom1').feature('dummyCircle');
circle.set('type', 'solid');
circle.set('base', 'center');
circle.set('pos', [0 0]);
circle.set('r', 'd/2');

%some settings for the rectangle option
sepInGroup = min(abs(diff(y(electrode==max(electrode))))); %this is true most of the cases
buildingRectangle = false;

%Loop over all structures
for elect = unique(electrode)  
    j = 0;
    names = {};
    inds = find(electrode==elect);
    for i = 1:length(inds)
        ind = inds(i);
        j = j + 1;
        %generate and store a name
        name = ['c',num2str(ind)];
        names{j} = name;
        %copy the dummy circle and place it in the desired position
        model.component('comp1').geom('geom1').feature.duplicate(name, 'dummyCircle');
        model.component('comp1').geom('geom1').feature(name).set('pos', [y(ind), x(ind)]);
        if rectangle
            if ~buildingRectangle
                %create rectangle (wires must be positioned parallel 
                %to x-axis)
                j = j + 1;                
                name = ['r',num2str(ind)];
                names{j} = name;
                model.component('comp1').geom('geom1').create(name, 'Rectangle');
                blk = model.component('comp1').geom('geom1').feature(name);
                blk.set('type', 'solid');
                blk.set('base', 'corner');
                blk.set('pos', [y(ind),x(ind)-d/2]);
                xstart = y(ind);
                buildingRectangle = true;
            else 
                %check if the separation is larger than the
                %groupseparation (and allow for some numerical err)
                if i == length(inds) || abs(y(ind) - y(inds(i + 1))) > sepInGroup*1.001
                    %we go into a next group, finish the rectangle by
                    %setting the size.
                    blk.set('size', [y(ind)-xstart,d]);
                    buildingRectangle = false;
                end              
            end
        end
    end
    %create a union of all wires of the current electrode
    uni = model.component('comp1').geom('geom1').create(['Structure'...
        ,num2str(elect)], 'Union');
    uni.set('selresult',true); 
    uni.selection('input').set(names);
end


%delete dummy circle
model.component('comp1').geom('geom1').feature.remove('dummyCircle');


%Add materials copper and air
%standard definition of copper and air.
model.component('comp1').material.create('mat1', 'Common');  model.component('comp1').material('mat1').propertyGroup('def').func.create('eta', 'Piecewise');  model.component('comp1').material('mat1').propertyGroup('def').func.create('Cp', 'Piecewise');  model.component('comp1').material('mat1').propertyGroup('def').func.create('rho', 'Analytic');  model.component('comp1').material('mat1').propertyGroup('def').func.create('k', 'Piecewise');  model.component('comp1').material('mat1').propertyGroup('def').func.create('cs', 'Analytic');  model.component('comp1').material('mat1').propertyGroup('def').func.create('an1', 'Analytic');  model.component('comp1').material('mat1').propertyGroup('def').func.create('an2', 'Analytic');  model.component('comp1').material('mat1').propertyGroup.create('RefractiveIndex', 'Refractive index');  model.component('comp1').material('mat1').propertyGroup.create('NonlinearModel', 'Nonlinear model');  model.component('comp1').material('mat1').label('Air');  model.component('comp1').material('mat1').set('family', 'air');  model.component('comp1').material('mat1').propertyGroup('def').func('eta').set('arg', 'T');  model.component('comp1').material('mat1').propertyGroup('def').func('eta').set('pieces', {'200.0' '1600.0' '-8.38278E-7+8.35717342E-8*T^1-7.69429583E-11*T^2+4.6437266E-14*T^3-1.06585607E-17*T^4'});  model.component('comp1').material('mat1').propertyGroup('def').func('eta').set('argunit', 'K');  model.component('comp1').material('mat1').propertyGroup('def').func('eta').set('fununit', 'Pa*s');  model.component('comp1').material('mat1').propertyGroup('def').func('Cp').set('arg', 'T');  model.component('comp1').material('mat1').propertyGroup('def').func('Cp').set('pieces', {'200.0' '1600.0' '1047.63657-0.372589265*T^1+9.45304214E-4*T^2-6.02409443E-7*T^3+1.2858961E-10*T^4'});  model.component('comp1').material('mat1').propertyGroup('def').func('Cp').set('argunit', 'K');  model.component('comp1').material('mat1').propertyGroup('def').func('Cp').set('fununit', 'J/(kg*K)');  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('expr', 'pA*0.02897/R_const[K*mol/J]/T');  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('args', {'pA' 'T'});  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('dermethod', 'manual');  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('argders', {'pA' 'd(pA*0.02897/R_const/T,pA)'; 'T' 'd(pA*0.02897/R_const/T,T)'});  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('argunit', 'Pa,K');  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('fununit', 'kg/m^3');  model.component('comp1').material('mat1').propertyGroup('def').func('rho').set('plotargs', {'pA' '0' '1'; 'T' '0' '1'});  model.component('comp1').material('mat1').propertyGroup('def').func('k').set('arg', 'T');  model.component('comp1').material('mat1').propertyGroup('def').func('k').set('pieces', {'200.0' '1600.0' '-0.00227583562+1.15480022E-4*T^1-7.90252856E-8*T^2+4.11702505E-11*T^3-7.43864331E-15*T^4'});  model.component('comp1').material('mat1').propertyGroup('def').func('k').set('argunit', 'K');  model.component('comp1').material('mat1').propertyGroup('def').func('k').set('fununit', 'W/(m*K)');  model.component('comp1').material('mat1').propertyGroup('def').func('cs').set('expr', 'sqrt(1.4*R_const[K*mol/J]/0.02897*T)');  model.component('comp1').material('mat1').propertyGroup('def').func('cs').set('args', {'T'});  model.component('comp1').material('mat1').propertyGroup('def').func('cs').set('dermethod', 'manual');  model.component('comp1').material('mat1').propertyGroup('def').func('cs').set('argunit', 'K');  model.component('comp1').material('mat1').propertyGroup('def').func('cs').set('fununit', 'm/s');  model.component('comp1').material('mat1').propertyGroup('def').func('cs').set('plotargs', {'T' '273.15' '373.15'});  model.component('comp1').material('mat1').propertyGroup('def').func('an1').set('funcname', 'alpha_p');  model.component('comp1').material('mat1').propertyGroup('def').func('an1').set('expr', '-1/rho(pA,T)*d(rho(pA,T),T)');  model.component('comp1').material('mat1').propertyGroup('def').func('an1').set('args', {'pA' 'T'});  model.component('comp1').material('mat1').propertyGroup('def').func('an1').set('argunit', 'Pa,K');  model.component('comp1').material('mat1').propertyGroup('def').func('an1').set('fununit', '1/K');  model.component('comp1').material('mat1').propertyGroup('def').func('an1').set('plotargs', {'pA' '101325' '101325'; 'T' '273.15' '373.15'});  model.component('comp1').material('mat1').propertyGroup('def').func('an2').set('funcname', 'muB');  model.component('comp1').material('mat1').propertyGroup('def').func('an2').set('expr', '0.6*eta(T)');  model.component('comp1').material('mat1').propertyGroup('def').func('an2').set('args', {'T'});  model.component('comp1').material('mat1').propertyGroup('def').func('an2').set('argunit', 'K');  model.component('comp1').material('mat1').propertyGroup('def').func('an2').set('fununit', 'Pa*s');  model.component('comp1').material('mat1').propertyGroup('def').func('an2').set('plotargs', {'T' '200' '1600'});  model.component('comp1').material('mat1').propertyGroup('def').set('thermalexpansioncoefficient', '');  model.component('comp1').material('mat1').propertyGroup('def').set('molarmass', '');  model.component('comp1').material('mat1').propertyGroup('def').set('bulkviscosity', '');  model.component('comp1').material('mat1').propertyGroup('def').set('relpermeability', {'1' '0' '0' '0' '1' '0' '0' '0' '1'});  model.component('comp1').material('mat1').propertyGroup('def').set('relpermittivity', {'1' '0' '0' '0' '1' '0' '0' '0' '1'});  model.component('comp1').material('mat1').propertyGroup('def').set('dynamicviscosity', 'eta(T)');  model.component('comp1').material('mat1').propertyGroup('def').set('ratioofspecificheat', '1.4');  model.component('comp1').material('mat1').propertyGroup('def').set('electricconductivity', {'0[S/m]' '0' '0' '0' '0[S/m]' '0' '0' '0' '0[S/m]'});  model.component('comp1').material('mat1').propertyGroup('def').set('heatcapacity', 'Cp(T)');  model.component('comp1').material('mat1').propertyGroup('def').set('density', 'rho(pA,T)');  model.component('comp1').material('mat1').propertyGroup('def').set('thermalconductivity', {'k(T)' '0' '0' '0' 'k(T)' '0' '0' '0' 'k(T)'});  model.component('comp1').material('mat1').propertyGroup('def').set('soundspeed', 'cs(T)');  model.component('comp1').material('mat1').propertyGroup('def').set('thermalexpansioncoefficient', {'alpha_p(pA,T)' '0' '0' '0' 'alpha_p(pA,T)' '0' '0' '0' 'alpha_p(pA,T)'});  model.component('comp1').material('mat1').propertyGroup('def').set('molarmass', '0.02897');  model.component('comp1').material('mat1').propertyGroup('def').set('bulkviscosity', 'muB(T)');  model.component('comp1').material('mat1').propertyGroup('def').addInput('temperature');  model.component('comp1').material('mat1').propertyGroup('def').addInput('pressure');  model.component('comp1').material('mat1').propertyGroup('RefractiveIndex').set('n', '');  model.component('comp1').material('mat1').propertyGroup('RefractiveIndex').set('ki', '');  model.component('comp1').material('mat1').propertyGroup('RefractiveIndex').set('n', {'1' '0' '0' '0' '1' '0' '0' '0' '1'});  model.component('comp1').material('mat1').propertyGroup('RefractiveIndex').set('ki', {'0' '0' '0' '0' '0' '0' '0' '0' '0'});  model.component('comp1').material('mat1').propertyGroup('NonlinearModel').set('BA', '(def.gamma+1)/2');  model.component('comp1').material('mat1').materialType('nonSolid');  model.component('comp1').material('mat1').set('family', 'air');
model.component('comp1').material.create('mat2', 'Common');  model.component('comp1').material('mat2').propertyGroup.create('Enu', 'Young''s modulus and Poisson''s ratio');  model.component('comp1').material('mat2').propertyGroup.create('linzRes', 'Linearized resistivity');  model.component('comp1').material('mat2').label('Copper');  model.component('comp1').material('mat2').set('family', 'copper');  model.component('comp1').material('mat2').propertyGroup('def').set('relpermeability', {'1' '0' '0' '0' '1' '0' '0' '0' '1'});  model.component('comp1').material('mat2').propertyGroup('def').set('electricconductivity', {'5.998e7[S/m]' '0' '0' '0' '5.998e7[S/m]' '0' '0' '0' '5.998e7[S/m]'});  model.component('comp1').material('mat2').propertyGroup('def').set('thermalexpansioncoefficient', {'17e-6[1/K]' '0' '0' '0' '17e-6[1/K]' '0' '0' '0' '17e-6[1/K]'});  model.component('comp1').material('mat2').propertyGroup('def').set('heatcapacity', '385[J/(kg*K)]');  model.component('comp1').material('mat2').propertyGroup('def').set('relpermittivity', {'1' '0' '0' '0' '1' '0' '0' '0' '1'});  model.component('comp1').material('mat2').propertyGroup('def').set('density', '8960[kg/m^3]');  model.component('comp1').material('mat2').propertyGroup('def').set('thermalconductivity', {'400[W/(m*K)]' '0' '0' '0' '400[W/(m*K)]' '0' '0' '0' '400[W/(m*K)]'});  model.component('comp1').material('mat2').propertyGroup('Enu').set('youngsmodulus', '110e9[Pa]');  model.component('comp1').material('mat2').propertyGroup('Enu').set('poissonsratio', '0.35');  model.component('comp1').material('mat2').propertyGroup('linzRes').set('rho0', '');  model.component('comp1').material('mat2').propertyGroup('linzRes').set('alpha', '');  model.component('comp1').material('mat2').propertyGroup('linzRes').set('Tref', '');  model.component('comp1').material('mat2').propertyGroup('linzRes').set('rho0', '1.72e-8[ohm*m]');  model.component('comp1').material('mat2').propertyGroup('linzRes').set('alpha', '0.0039[1/K]');  model.component('comp1').material('mat2').propertyGroup('linzRes').set('Tref', '298[K]');  model.component('comp1').material('mat2').propertyGroup('linzRes').addInput('temperature');  model.component('comp1').material('mat2').set('family', 'copper');  model.component('comp1').material('mat2').selection.geom('geom1', 1);  model.component('comp1').material('mat2').selection.geom('geom1', 0);  model.component('comp1').material('mat2').selection.geom('geom1', 2);


%make the geometry
model.component('comp1').geom('geom1').run;

%% set boundary conditions
j = 0;
for elect = unique(electrode)
    ind = find(electrode == elect,1);
    pot = U(ind);
    domains = mphgetselection(model.selection(['geom1_',...
        'Structure',num2str(elect),'_dom'])).entities;
    %set wires to material copper
    model.component('comp1').material('mat2').selection.set(domains);
    
    %create appropriate terminal or ground bc's
    if elect == 0
        gnd = model.component('comp1').physics('es').create('gnd1', 'Ground', 1);
        %find boundaries of the domains (to set ground to the bounds)
        bounds = mphgetadj(model,'geom1','boundary','domain',domains);
        gnd.selection.set(bounds);
    else
       j = j + 1;
       terminal =  model.component('comp1').physics('es'...
           ).create(['term',num2str(elect)], 'DomainTerminal', 2);
       terminal.selection.set(domains);
       terminal.set('TerminalType', 'Voltage');
       model.component('comp1').physics('es').feature(['term',...
           num2str(elect)]).set('TerminalName',num2str(j));
       terminal.set('V0', pot);
    end
    
end

%% Study details (standard stationary setup)
%parametric study to loop over the different ports to find maxwell
%capacitance
model.sol.create('sol1');
model.sol('sol1').study('std1');
model.sol('sol1').attach('std1');
model.sol('sol1').create('st1', 'StudyStep');
model.sol('sol1').feature('st1').set('study', 'std1');
model.sol('sol1').feature('st1').set('studystep', 'stssw');
model.sol('sol1').create('v1', 'Variables');
model.sol('sol1').feature('v1').set('control', 'stssw');
model.sol('sol1').create('s1', 'Stationary');
model.sol('sol1').feature('s1').create('p1', 'Parametric');
model.sol('sol1').feature('s1').feature('p1').set('preusesol', 'no');
model.sol('sol1').feature('s1').feature('p1').set('pcontinuationmode', 'no');
model.sol('sol1').feature('s1').feature('p1').set('control', 'stssw');
model.sol('sol1').feature('s1').create('fc1', 'FullyCoupled');
model.sol('sol1').feature('s1').feature('fc1').set('linsolver', 'dDef');
model.sol('sol1').feature('s1').feature.remove('fcDef');
model.sol('sol1').attach('std1');


%% Find capacitance
if length(unique(electrode))>2
    EGM4 = model.result.numerical.create('gmev4', 'EvalGlobalMatrix');
    EGM4.set('expr', 'root.comp1.es.Cinv');
    EGM4.set('trans', 'invmaxwellmutual');
    EGM4.set('outerdataseries', 'none');
    EGM4.label('Mutual capacitance (es, dset1)');
    EGM4.set('data', 'dset1');
    EGM4.set('descr', 'Mutual capacitance');
    EGM4.set('dataseries', 'average');
    model.result.table.create('tbl5', 'Table');
    model.result.table('tbl5').label('Mutual capacitance');
    EGM4.set('table', 'tbl5');
end

model.sol('sol1').runAll;
%if we have more than 2 terminals we can extract the mutual capacitance
%from the tables, otherwise we get it from a global expression.
if length(unique(electrode))>2
    EGM4.setResult;
    %mutual capacitance
    C = -diff(diag(model.result.table('tbl5').getReal));
else
    %maxwell capacitance
    C = 1/mphglobal(model,'es.Cinv11');
end


%% plot (optional)
if plot
    %% determine suitable bounds
    smin = 0;
    for i = 1:length(y)
        for j = 1:i-1
            %create a matrix with the separation between (centers of) wires
            s = sqrt((y(i)-y(j))^2 + (x(i)-x(j))^2);
            if s > smin
                smin = s;
            end
        end
    end
    %find suitable y bounds
    ymax = max(x) + smin;
    ymin = min(x) - smin;
    %find suitable x bounds
    xmax = max(y) + smin;
    xmin = min(y) - smin; 

    %create a plot
    model.result.create('pg1', 'PlotGroup2D');
    model.result('pg1').create('con1', 'Contour');
    model.result('pg1').feature('con1').set('expr', 'V');
    model.result('pg1').feature('con1').set('descr', 'Electric potential');
    model.result('pg1').run;
    mphplot(model,'pg1');
    xlim([xmin,xmax])
    ylim([ymin,ymax])
end

%% Check mesh option
if checkMesh
    %create a finer mesh    
    model.component('comp1').mesh.create('mesh2');
    model.component('comp1').mesh('mesh2').autoMeshSize(3);
    model.component('comp1').mesh('mesh2').run;
    
    %run the model with the finer mesh
    model.study('std1').feature('stssw').setIndex('mesh', 'mesh2', 1);
    model.sol('sol1').attach('std1');
    model.sol('sol1').runAll;
    
    %compare and print a result message
    if length(unique(electrode))>2
        EGM4.setResult;
        %differential capacitance
        Cfine = -diff(diag(model.result.table('tbl5').getReal));
    else
        %maxwell capacitance
        Cfine = 1/mphglobal(model,'es.Cinv11');
    end
    
    %compare and print a result message
    fprintf('The (max) mutual capacitance with a normal mesh is %.5e F/m, with a finer mesh %.5e F/m (%.4f%% difference) \n', ...
        max(C,[],'all'), max(Cfine,[],'all'), (max(C,[],'all')-max(Cfine,[],'all'))/max(C,[],'all')*100)
end


%% Check space influence (optional)
if checkSpace
    %increase each dimension by 10 percent
    model.study('std1').feature('stssw').setIndex('mesh', 'mesh1', 1);
    rect.set('size', [width*1.1,height*1.1]);
    model.sol('sol1').runAll;
    
    %find the capacitance matrix (>2 dimensions) or capacitance
    if length(unique(electrode))>2
        EGM4.setResult;
        %mutual capacitance
        Clarge = diag(model.result.table('tbl5').getReal);
        Clarge = Clarge(1)-Clarge(2);
    else
        %maxwell capacitance
        Clarge = 1/mphglobal(model,'es.Cinv11');
    end

    %compare and print a result message
    fprintf('The (max) mutual capacitance with the usual settings %.5e F/m, with a 10%% larger space %.5e F/m (%.4f%% difference) \n', ...
        max(C,[],'all'), max(Clarge,[],'all'), (max(C,[],'all')-max(Clarge,[],'all'))/max(C,[],'all')*100)
end
%% save file (optional)
if saveFile
    mphsave(model,'output.mph')
end
if plot
    mphgeom(model)
end
