function [lambda] = potentialToCenteredCharge2(y,x,U,d)
%POTENTIALTOCENTEREDCHARGE2 calculates the charge needed to achieve a
%defined potential difference in a wire structure.
%The wire structure consists of a number of infinetely long, finitely 
%thick cylindrical conductors, of which the field is approximated 
%using line charges centered on the conductor. 
%This function finds the associated line charges, given the potential 
%of each wire. This is done by inversion of the elastance matrix.
%
%
%Inputs:
% -x and y define the position of the center of the wires (1xn array)
% -U defines the potential of the wires (1xn array)
% -d defines the diameter of the wires (equal for all wires)
%Outputs the charge (in [C/m]) on all wires in an array.
%

%% Check of the input
if nargin < 4
    error('Not enough input arguments')
end

n = length(y);         %number of wires
e0 = 8.8541878128E-12; %permittivity (of vacuum)

%check if vectors have equal length
if length(y) ~= n || length(x) ~= n || length(U) ~= n
    error('Inconsistent number of wires (x, y and U have different dimensions)')
end

%% select the match points
%use np match points on the wires
npos = 16;
%we select the points with equal separation (not using the last point):
list = linspace(0,360,npos+1); 

%preallocation
xmeas = nan(length(y),npos);
ymeas = xmeas;

%calculate the position of the point
for i = 1:npos
    angle = list(i);
    xmeas(:,i) = y + cosd(angle)*d/2;
    ymeas(:,i) = x + sind(angle)*d/2;
end


%% construction of the separation matrix s
%This matrix stores the separation between centers of wires. The 
%separation between the center of wire i and the potential measurement 
%point k of wire j is stored in s(i,j,k). 

s = nan(n,n,npos); %preallocation

%find the center position
centerx = mean(y);
centery = mean(x);
warned = false;
for k = 1:npos
    for i = 1:length(y)
        for j = 1:length(x)
            %Calculate the separation
            s(i,j,k) = sqrt((y(i)-xmeas(j,k))^2 + (x(i)-ymeas(j,k))^2);
            if s(i,j,k) < .501*d && i~= j && ~warned
                warned = true;
                warning('Some wires are very close, this can lead to a singular matrix ')
            end                
        end
    end
end


%% Determination of the difference paths.
%It does not matter which differences are used, so we use neighbouring
%pairs (in index).
w1 = 1:length(y)-1;
w2 = 2:length(y);

%% construction of the potential vector
%this vector gives the potential differences between each neighbouring 
%pair of wires (in terms of index). The last 0 is to generate a last 
%equation in the system: total charge = 0.
phi = [(U(w1)-U(w2))'; 0];

%% constuction of the capacitance matrix
M = zeros(n,n);
for i = 1:length(w1)
   for j = 1:n
       for k = 1:npos
           %average the potential of the measurement points
           M(i,j) = M(i,j) + log(s(j,w2(i),k)/s(j,w1(i),k)) / npos;
       end
   end
end
%Finish the matrix by division by the common constant
M = M/(2*pi()*e0);
%enter a last row of ones for the last equation (net charge = 0)
M(n,1:n) = ones(1,n);

%find the charge vector by inversion of the M matrix.
lambda = M\phi;

end

