function [x,y,phi,alpha,lambda,theta,k] = neoHookean(L,H,QmuAc,PmuAc,varargin)
%NEOHOOKEAN Numerically solves the deformation of beams
%of the timoshenko type with a neo-Hookean material model. 
%As boundary conditions use that the beam is clamped on one side and 
%has a fixed angle and a horizontal and vertical loading on the other
%side. 
%
%This script implements the equations for the model derived in section
%2.3 of the accompanying paper. These are based upon the model derived
%by He. et al. (2018) (https://doi.org/10.1007/s11012-018-0905-4)
%
%The numerical appoach to solve the system of equations (eq. (3)) used 
%here is similar to that described by He et al. and is as follows:
% 1. The functions phi, alpha, and lambda (of x) are discretized for
%    50 points on the beam
% 2. An initial guess is made for alpha and lambda (on these 50 points)
% 3. With this guess of alpha and lambda, and the boundary conditions,
%    equation (3c) is solved for phi using a numerical solver based
%    upon the Trust-Region-Dogleg Algorithm (the Matlab fsolve()
%    function) Subsequently, phi' is estimated using a central
%    difference scheme. 
% 4. The estimate of phi is then substituted in equations (3a) and
%    (3b). This gives 2 equations (for each point on the beam) with
%    two unknowns: alpha and lambda. This system of equations is then
%    solved using the same solver as in 3. 
% 5. Steps 3 and 4 are repeated until the convergence criterion: the
%    sum of the differences in alpha and lambda on the 50 points is
%    less than 0.001. 
% 6. The deformation in x and y coordinates is found by summing the
%    terms of delta u_0 and delta v_0 respectively. These can be found
%    from lambda and theta:
%         delta u_0 = delta x lambda cos(theta)
%         delta v_0 = delta x lambda sin(theta)
%
%Inputs:
% - L: Length of the beam
% - H: Height of the beam 
% - QmuAc: shear loading on the end of the beam (in N) divided by the
%          sheaar modulus mu and the crossectional area A_c.
% - PmuMc: Normal loading on the end of the beam (in N) divided by the
%          sheaar modulus mu and the crossectional area A_c.
%Optional inputs:
% - 'reduceOutput': Reduces the output for use in optimization
%          functions: only the maximum y deviation will be outputted 
%          as first output
% - 'guessAlphaLambda': followed by two arrays of length 50 can be 
%          used to replace the initial guess of alpha and lambda. This
%          is especially useful when calculating a large number of 
%          deformations in a loop to use the previous result as an 
%          initial guess to reduce computation time.
%
%Outputs the deformation of the geometrical midplane (in x and y
%coordinates) and the angles phi, alpha, theta togehter with lambda
%and k, the first derivative of phi.



npoints = 50;   %number of points on the rod

%initial guess for the deformation
alpha(1,:) = ones(1,npoints)*0.000000001; 
lambda(1,:) = ones(1,npoints);
dx = L/(npoints-1); 
phi = zeros(1,npoints);

i=0;
simpleOut = false;
%analyze optional arguments
while i < length(varargin)
    i = i + 1;
    in = varargin{i};
    if strcmp(in,'reduceOutput')
        %option to only return maximum y position (for use in
        %optimization function)
        simpleOut = true;
    elseif strcmp(in,'guessAlphaLambda')
        %overwrite default guesses if needed
        i = i + 1;
        alpha(1,:) = varargin{i};
        i = i + 1;
        lambda(1,:) = varargin{i};
    else
        warning(['Unknown option ', in,' is ignored'])
    end
end  

change = Inf;
%turn off display of the solver
options = optimoptions('fsolve','Display','off');
i = 0;
%paramater that slows the change in parameters to increase chances of
%convergence
slowparam = .05; 

%In this loop we first solve for phi. then we use phi to solve for
%better estimates of alpha and lambda. This is repeated until the
%change between iteration is small  (or until we notice that we
%diverge). 
iterations = 0;

%check if we converged or have a divergent task
while change > 1e-2 && iterations < 200 && (change < 100 || iterations < 5)
    iterations = iterations + 1;
    
    %solve for phi
    phi = fsolve(@findPhi,phi,options);
    
    %make an estimate for k (= phi')
    k=[(phi(2)-phi(1))/dx, derivative(phi), (phi(end)-phi(end-1))/dx];
    
    %make new estimates for alpha and lambda
    al = fsolve(@findAlphaLambda,ones(1,2*npoints),options);
    %alpha and lambda are together in one array. Separate them:
    nextalpha = al(1:end/2);
    nextlambda = al(end/2+1:end);
    
    %check how much we changed in this step (to check convergence)
    change = sum(abs(alpha-nextalpha))+sum(abs(lambda-nextlambda));
    
    alpha = nextalpha  * slowparam + alpha * (1-slowparam);
    lambda = nextlambda* slowparam + lambda* (1-slowparam);
    i = i + 1;
end
%check if we have converged
if change > 1e-2
    x = NaN;
    y = NaN;
    phi = NaN;
    theta = NaN;
    k = NaN;
    fprintf('Solution of the neoHookean() model did not converge')
    return
end

%calculate the deformation of the rod in x and y coordinates
theta = alpha + phi;
twoPointAverage = @(array) (array(1:end-1) + array(2:end))/2;
x = [0,movsum(cos(twoPointAverage(theta)).*...
        twoPointAverage(lambda).*dx,[npoints,0])];
y = [0,movsum(sin(twoPointAverage(theta)).*...
        twoPointAverage(lambda).*dx,[npoints,0])];

if simpleOut
    %reduced output for use in optimization functions
    x = y(end);
    y = NaN;
    phi = NaN;
    theta = NaN;
    k = NaN;
    return
end

%function inputs for fsolve in line with equation 3b
function zero = findPhi(phi)
    c = 1/3 .* H^2 ./ (lambda .* cos(alpha)).^6;
    err = 1./lambda(2:end-1) .* derivative(c) .* derivative(phi) ...
        + 1./lambda(2:end-1) .* c(2:end-1) .* derivative2(phi) ...
        +QmuAc .* cos(alpha(2:end-1) + phi(2:end-1)) ...
        -PmuAc .* sin(alpha(2:end-1) + phi(2:end-1));

    %function to be minimized, which in addition incorporates the
    %boundary conditions.
    zero = [phi(1),phi(end),err];
end

%use eq. (3a) and (3c) for new alpha and lambda
function zero = findAlphaLambda(al)
    
    nextA = al(1:end/2);
    nextL = al(end/2+1:end);
    a = - 1./(lambda.*cos(alpha)).^3 ...
        - k.^2.*H^2./(lambda.*cos(alpha)).^7 - QmuAc .* sin(phi);
    zero = [nextL.*cos(nextA) - 1./(lambda.*cos(alpha)).^3 ...
        - k.^2.*H^2./(lambda.*cos(alpha)).^7 ...
        - QmuAc .* sin(phi) - PmuAc .* cos(phi) , ... 
        nextL.*sin(nextA) - QmuAc .* cos(phi) + PmuAc .*sin(phi)];
end

%functions to find the numerical approximation of the first and second
%derivative:
function ddx = derivative(y)
    %central difference scheme
    ddx = (y(3:end) - y(1:end-2)) ./ (2*dx);
end
function ddxdx = derivative2(y)
    %numerical approximation of the second derivative
    ddxdx = (y(3:end) - 2.*y(2:end-1) + y(1:end-2)) ./ dx^2;
end

    

end
