function [C,q1] = findCapacitance(U,lambda,varargin)
%FINDCAPACITANCE Calculates the capacitance of a given wire structure
%This function uses the simple equation C = q/V, this only works for a
%wire structure consisting of two electrode with each a single potential. 
%This means that only two different potentials can be present. 

%Required inputs: 
% - U vector specifies the potential on all wires
% - lambda is a vector containing the charges on the wires
%Optional argument: msg - if set to true, the capacitance will be
%printed in the command window.

% analyze optional input arguments;
i = 0;  
msg = false;
multiElectrode = false;

while i < length(varargin)
    i = i + 1;
    in = varargin{i};
    if islogical(in) || strcmp(in,'msg')
        %backwards compatibility, only true was used to indicate msg.
        if islogical(in) && ~in
            %do nothing
        else
            msg = true;
        end
    elseif strcmp(in,'multiElectrode')
        i = i + 1;
        multiElectrode = true;
        electrode = varargin{i};        
    else
        warning(['Unknown option ', in,' is ignored'])
    end
end 

%% Check input arguments
if nargin < 2
    error('Not enough input arguments')
end

%% Calculate capacitance
%check if there are 2 structures with each equal potential on all wires
uU = unique(U);
if ~multiElectrode
    q1 = sum(lambda(U == uU(1)));
else
    uElectrode = 1:max(electrode);
    q1 = sum(lambda'.*(electrode == uElectrode'),2);
end

if length(uU) ~= 2
    error('This function only calculates capacitance between structures with a maximum of 2 unique potentials')
end
%check if the total charge is zero (which is required for the simple
%capacitance calculation
if abs(sum(lambda)) > abs(1e-10*q1) %allow for some numerical error
    error('Total charge is nonzero')
end



%% calculate the capacitance
if ~multiElectrode
    C = abs(q1/(uU(1)-uU(2)));
else
    C = (q1(1)-q1(2))/abs(uU(1)-uU(2));
end

%optionally print to command window.
if msg
    fprintf('The capacitance is %.2e F/m. \n',C)
end


        
end

