%% Demo for use of the models and optimization tools
% This script introduces the use of the crucial models and tools used
% for optimization of a capacitive shear force sensor. The theory that
% underpins the methods used here is discussed in the paper. 
% This demo file intends to show the purpose of the most important
% functions that are included. It is advised to use a breakpoint and
% go step by step to see how the functions work.
% The models introduced here are used in the OptimizationApp, which is
% used to optimize the design of the sensor. 


clear;      %clear workspace
close all;  %close open figures
clc;        %clear the command window

%% DEFINITION OF A WIRE STRUCTURE

%First, a wire structure is defined. The parameters are the same as in
%figure 2 in the paper.

nGroups = 3;    %Number of wire groups
syGroups = 4;   %Horizontal distance between wire groups [mm]
nInGroup = 3;   %Number of wires in a group 
syInGroup = 0.6;%Distance between wires in a group [mm]
d = 0.4;        %Diameter of the wires
Sx = 1.2;       %Vertical distance between top and bottom structure [mm]

%For other functions, it is convenient to rephrase the structure in
%just x and y coordinates and a potential. For the potential, we
%usually set the bottom structure to -1V and the top structure to +1V.
%For this, we can use the createWireStructure() function:
[ypos,xpos,U] = createWireStructure(nGroups,nGroups,nInGroup,syGroups,syInGroup,Sx);
figure('units','normalized','outerposition',[0 7/13 1/3 6/13])
plotWires(ypos,xpos,d)
axis equal
%% CAPACITIVE MODEL USING APPROXIMATION
%After the structure has been defined, the center wire approximation 
%can be used to find the capacitnace of the wire structure.
%The pontentialToCenteredCharge2() function uses this approximation to
%find the charge on the different wires from the potential:
lambda = potentialToCenteredCharge2(ypos,xpos,U,d);

%Now that we know the charge on the different wires, the capacitance
%between the two electrodes can be found. This can be done using the
%findCapacitance() function:
C = findCapacitance(U,lambda,'msg');

%Often, we are interested to find the capacitance-displacement
%behavior of a certain wire structure. The capacitanceDisplacement()
%function uses the method we demonstrated above to find the
%capacitance for wire structures where the top structure is gradually
%displaced. 
displacement = linspace(-6.5,6.5,200);
Cfig = figure('units','normalized','outerposition',[1/3 7/13 1/3 6/13]);
C_list = capacitanceDisplacment(ypos,xpos,U,0.4,displacement,'plot');

%To find the best initial displacement y0, we want to have the maximum 
%slope of the graph. The case of the maximum derivative is already 
%highlighted, but it is also possible to only plot the derivative:
dCfig = figure('units','normalized','outerposition',[2/3 7/13 1/3 6/13]);
C_list = capacitanceDisplacment(ypos,xpos,U,0.4,displacement,'plot','dCdy','y0');

%We can extend our analysis to the multielectrode case. (For
%optimization of y0 with multiple electrodes, only dCdy is interesting 
%because differential capactiance will be 0 because of symmetry.)

%use the capacitanceDisplacment function:
figure(dCfig)
C_list = capacitanceDisplacment(ypos,xpos,U,0.4,[],'plot','multiElectrode','dCdy','y0');
% C_list = capacitanceDisplacment(ypos,xpos,U,0.4,[],'plot','multiElectrode','optimizey0');

Ch = get(gca,'Children');
Ch(2).YData = Ch(2).YData*2; %To compare our results we can multiply the single electrode case by 2
legend('2*Single electrode','Multielectrode')

%To get even more insight in what happens when the top of a wire
%structure is displaced, the capacitanceDisplacement() function can be
%used to make an animation
% C_list = capacitanceDisplacment(ypos,xpos,U,0.4,displacement,'plot','animation','paperLayout');

%(It may take a while to make this animation, to watch it at a higher
%speed, it is also saved as vid.avi in the current folder).

%A similar animation can be made for the multiElectrode case:
C_list = capacitanceDisplacment(ypos,xpos,U,0.4,[],'plot','animation','multiElectrode','potField','y0','dCdy','paperLayout');

%% FE METHODS TO MODEL THE CAPACITIVE BEHAVIOR
%For FE methods, COMSOL Multiphysics is used. To find simulation
%results from Matlab, it is required to have a connection between
%Matlab and Comsol. This is achieved using Comsol Livelink for Matlab.
%Ensure that you have such a connection to Comsol to use these
%methods. (FE methods are mainly used for verification, therefore it
%is very well possible to use the rest of the models and the
%optimization tool without using FE methods.)

%After the wire structure is defined, the FEMwirestructure() function
%can be used to do a FE simulation for infinitely long wire
%structures.
%Optionally, we can check the influence of the mesh on the result:
C_FEM = FEMwireStructure(ypos,xpos,U,U,d,'checkMesh','saveFile');


%The function can also be used to find the capacitance between
%structures of rounded plates. Here the space between the conductive
%wires in a group is filled to form rounded rectangles. A plot of the
%geometry is made to see the exact structure. 
FEMfig = figure('units','normalized','outerposition',[0 1/13 1/3 6/13]);
C_FEM_rectangle = FEMwireStructure(ypos,xpos,U,U,d,'rectangle','plot')
xlim([-8,8]) %zoom in on the plot such that the structure can be seen
ylim([-8,8])

%The FEMwireStructure3() function can be used to perform a
%three-dimensional FE simulation, so where the structures have finite
%length. The use is similar to the regular FEMwireStructure()
%function.
L = 22; %length in mm
C_FEM3 = FEMwireStructure3(ypos,xpos,U,U,d,L,'checkSpace');
%To compare to the infinite FEM, we should divide by the length to
%have unit F/m again:
fprintf('That is %.4e F/m\n',C_FEM3/(L/1000))


%% MECHANICAL MODEL USING HYPERELASTIC BEAM MODEL
%For the beam model of a Timoshenko type beam of a neo-hookean
%material can be used to predict the deformation. The main interest is
%in how the end of the beam displaces subject to normal and shear
%stresses, because that is how the top of the capacitive structure
%displaces. The model described in the paper is implemented in the
%neoHookean() function:


mu = 1.32e8;   %Initial shear modulus (Armadillo)
E = 2*(1.5)*mu;%Youngs modulus
Q = 140e3*3;   %Shear pressure [Pa]
P = 0;         %normal pressure [Pa]
L = 1;         %length of thse beam 
H = 0.2227;    %Width of the beam

%use the neoHookean() function to find the deformation of the beam:
[x,y,phi,alpha,lambda,theta,k] = neoHookean(L,H,Q/mu,P/mu);
MechFig = figure('units','normalized','outerposition',[1/3 1/13 1/3 6/13]);
plot(y,x) %show the deformed midplane.
title('Deformed midplane element')
xlabel('x')
ylabel('y')

axis equal

%In optimization, often the same deformation needs calculation. As the
%numerical solver in neoHookean is not very fast, it is advantageous
%to save the modeling results for later use. This is done using the
%findDeformation() function: If possible, it interpolates from the
%known deformations stored in the DeformationData.mat file. If this is
%not possible, the deformation is calcuated using neoHookean() and
%saved into the datafile (using the saveDeformation() function). 
[u,v] = findDeformation(H/L,P/mu,Q/mu,1,0,10)
figure(MechFig)
hold on 
plot(v,1+u,'*')
hold off

%% FE METHODS FOR MECHANICAL MODELING
%For FE methods, COMSOL Multiphysics is used. To find simulation
%results from Matlab, it is required to have a connection between
%Matlab and Comsol. This is achieved using Comsol Livelink for Matlab.
%Ensure that you have such a connection to Comsol to use these
%methods.
%
%The FE simulations for finding the deformation of a beam are
%performed using the mechanicalBeamFem() function:
Q = 160e3*3;   %Shear pressure [Pa]
P = 0;         %normal pressure [Pa]
L = 1;         %length of the beam
H = 0.2227;    %Width of the beam
[list,dispx, dispy] = mechanicalBeamFem(L,H,Q,P,'Q',Q)
figure(MechFig)
hold on
plot(dispx,1+dispy,'x') %this can be plotted onto the deformation 
                        %results from the other model
hold off

%Also 3D beams can be used in the mechanicalBeamFem() function using 
%the additional '3D' argument and 'WS-ratio' to set the width of the
%beam in the z-direction:

W = 2; %width of the beam
[list,dispx, dispy] = mechanicalBeamFem(L,H,Q,P,'Q',Q,'3D','WS-ratio',W/L)
figure(MechFig)
hold on
plot(dispx,1+dispy,'o') %this can be plotted onto the deformation 
                        %results from the other model
hold off

%The FE simulations for the complete sensor structure are done in
%Comsol Multiphysics directly. (Without a Livelink). Therefore no
%matlab scripts are available here. The model is included as
%'3D-sensor parameter mechanics.mph'

%% COMBINING MECHANICAL AND ELECTROSTATIC MODELS
%The capacitanceDisplacment function that was used to find the
%capacitance due to a displacment can also used to model the
%capacitance due to stresses. To model the stress to displacement, the
%hyperelastic beam model is used. When the displacment is known, the
%capacitance can be found like before. Note that now also small
%displacements in the y direction are present and that the range of
%displacements is usually smaller. Therefore, it is wise to optimize
%the initial displacement y0 first, this can be done by parsing the
%'optimizey0' input to capacitanceDisplacement:
CombFig = figure('units','normalized','outerposition',[2/3 1/13 1/3 6/13]);
C_list = capacitanceDisplacment(ypos,xpos,U,0.4,[],...
    'plot','optimizey0','modelMechanics',[],[],[],[],[],'inBlock');

%Combine with multiElectrode
C_list = capacitanceDisplacment(ypos,xpos,U,0.4,[],...
    'plot','optimizey0','modelMechanics',[],[],[],[],[],'inBlock','multiElectrode');
%Of course, the single electrode has a base capacitance, whilst the
%multiElectrode starts from a symmetric case, so without any
%capacitance difference. To compare the graphs we subtract the base 
%capacitance from the first graph, also for better comparisson we can 
%multiply the single electrode case by 2:
Ch = get(gca,'Children');
Ch(2).YData = (Ch(2).YData-Ch(2).YData(1))*2;  
legend('2*(Single electrode-base)','Multielectrode','Location','southeast')



%% OPTIMIZATION
%The Matlab models are used for optimization of the sensor design.
%This is done using a custom Matlab app. 
%This app uses the models derived here to perform parameter sweeps on
%the design. This helps to seek a (locally) optimal design. 
%FE models can optionally be used to verify the results (if this is
%not used, a livelink connection to Comsol is not strictly needed to
%use the tool). 
%Further details on how to use the tool can be found in the tool
%itself by hovering the blue i symbols in the app. 

OptimizationApp %launch the app.




