function [absError,relError] = checkPotential(xpos,ypos,d,U,lambda,msg)
%CHECKPOTENTIAL checks the potential at 360 points on the wires to 
%check how well the boundary conditions are matched.
%
%Inputs
%-xpos and ypos define the postions of the wires
%-d defines the diameter of all wires
%-U defines the desired potential on each wire
%-lambda contains the charge on each wire
%-[optional] if msg is set to false, no conclusion message will be 
%printed
%Returns maximal absolute and relative error in the potential 
%(in volts and fraction of maximal difference). This is also printed 
%in the command window if msg is not set to false.


%% Check given input
if nargin < 5
    error('Not enough input arguments')
end
if length(xpos) ~= length(ypos) || length(ypos) ~= length(lambda) || length(lambda) ~= length(U)
    error('Inconsistent number of wires (xpos, ypos, U and/or lambda have different dimensions)')
end


e0 = 8.8541878128E-12; %permittivity (of vacuum)

%% Select points of interest
X = nan(length(xpos),360);
Y = X;
theta = linspace(0,2*pi(),360);
for i = 1:length(xpos)
    X(i,:) = xpos(i) + cos(theta)*d/2;
    Y(i,:) = ypos(i) + sin(theta)*d/2;
end

%% Calculate the potential at the points of intest
%loop over all wires
phi = zeros(size(X));
for i = 1:length(xpos)
    a = xpos(i);
    b = ypos(i);
    q = lambda(i);
    %find potential of current wire
    phi1 = q./(2*pi()*e0)*log(sqrt((X-a).^2+(Y-b).^2));
    %superposition on top of calculated potential
    phi = phi + phi1;
end

%% Find errors
absError = max( max(phi,[],2) - min(phi,[],2) );
relError = absError/(max(U)-min(U));
%check wheter we need to print
if nargin == 6
    if msg == false
        return
    end
end
%pritn conclusion message
fprintf('The maximum potential difference is %.3f mV (%.2f %% of maximal potential difference).\n',absError*1000,relError*100)






