function [X,Y,phi] = chargeToPotential(xpos,ypos,lambda,xmin,xmax,ymin,ymax,n)
%CHARGETOPOTENTIAL Finds the potential field of a number of line charges.
%This is supposed to be the desired input for a contour/surf plot, as the
%results are for a grid in the xy-plane. 
%Note: In this function the role of x and y is flipped. 

%Inputs
%-xpos and ypos define the postion of the line charges
%-lambda defines the line charge that each wire carries.
%Optional:
%-xmin,xmax,ymin,ymax to set the grid boundaries
%-n to set the number of points in each direction
%
%Outputs an X,Y meshgrid with potential values (phi) in a matrix.

%% check validity of input
if nargin < 3
    error('Not enough input arguments')
end
if length(xpos) ~= length(ypos) || length(ypos) ~= length(lambda)
    error('Inconsistent number of wires (xpos, ypos and lambda have different dimensions)')
end
e0 = 8.8541878128E-12; %permittivity (of vacuum)

%% set alternative values for unset inputs
if nargin<8
    n = 500;
    if nargin < 7
        %find the minimum separation
        smin = 0;
        for i = 1:length(xpos)
            for j = 1:i-1
                %create a matrix with the separation between (centers of) wires
                s = sqrt((xpos(i)-xpos(j))^2 + (ypos(i)-ypos(j))^2);
                if s < smin
                    smin = s;
                end
            end
        end
        %set default y bounds
        ymax = max(ypos) + smin*4;
        ymin = min(ypos) - smin*4;
        if nargin < 5
            %set default x bounds
            xmax = max(xpos) + smin*4;
            xmin = min(xpos) - smin*4; 
        end
    end
end

%select suitable number of points in both directions. Use the inputted n
%for the smallest dimension.
if xmax-xmin < ymax-ymin
    nx = n;
    ny = round(n*(ymax-ymin)/(xmax-xmin));
else
    ny = n;
    nx = round(n/(ymax-ymin)/(xmax-xmin));
end

%% create x and y arrays and an X,Y meshgrid
x = linspace(xmin,xmax,n);
y = linspace(ymin,ymax,n);
[X,Y] = meshgrid(x,y);
%preallocate phi with zeros:
phi = zeros(n,n);

%% find potential field
for i = 1:length(xpos)
    a = xpos(i);
    b = ypos(i);
    q = lambda(i);
    %find potential of current wire
    phi1 = -q./(2*pi()*e0)*log(sqrt((X-a).^2+(Y-b).^2)./sqrt(a.^2+b.^2));
    %superposition on top of calculated potential
    phi = phi + phi1;
end

