import { defineStore } from 'pinia';
import { promiseTimeout } from '@vueuse/core';

interface Notification {
    id: string;
    type: 'error' | 'success';
    message: string;
}

interface State {
    bags: { [x: string]: { [x: string]: string[] } };
    notifications: Notification[];
}

export const useValidationStore = defineStore('validation', {
    state: (): State => ({
        bags: {},
        notifications: [],
    }),
    getters: {
        error:
            ({ bags }) =>
            (name: string, scope = 'default') => {
                if (scope in bags && name in bags[scope]) {
                    return bags[scope][name][0];
                }
                return null;
            },
    },
    actions: {
        addBag(errors: { [x: string]: string[] }, scope = 'default') {
            if (scope in this.bags) {
                this.bags[scope] = errors;
            } else {
                this.bags = { ...this.bags, [scope]: errors };
            }
        },
        async addNotification(message: string, type: 'error' | 'success' = 'error') {
            const id = Math.random().toString().substring(2, 9);
            this.notifications.unshift({ id, type, message });
            await promiseTimeout(5000);
            this.removeNotification(id);
        },
        removeBag(scope = 'default') {
            if (scope in this.bags) {
                delete this.bags[scope];
            }
        },
        removeError(name: string, scope = 'default') {
            if (scope in this.bags && name in this.bags[scope]) {
                delete this.bags[scope][name];
            }
            if (scope in this.bags && Object.keys(this.bags[scope]).length < 1) {
                delete this.bags[scope];
            }
        },
        removeNotification(id: string) {
            this.notifications = this.notifications.filter(
                (notification) => notification.id !== id,
            );
        },
    },
});
