import { Chart, ScriptableTooltipContext } from 'chart.js';

export function useTooltip() {
    const getOrCreateTooltip = (chart: Chart<'line'>) => {
        let tooltipElement = chart.canvas.parentNode?.querySelector('div');

        if (!tooltipElement) {
            tooltipElement = document.createElement('div');

            tooltipElement.classList.add(
                'absolute',
                'rounded-md',
                'border',
                'bg-white',
                'px-4',
                'py-3',
                'text-sm',
                '-translate-x-1/2',
                'transition-all',
            );

            tooltipElement.style.opacity = '1';

            chart.canvas.parentNode?.appendChild(tooltipElement);
        }

        return tooltipElement;
    };

    const tooltipHandler = (context: ScriptableTooltipContext<'line'>) => {
        const { chart, tooltip } = context;
        const tooltipElement = getOrCreateTooltip(chart);

        if (tooltip.opacity === 0) {
            tooltipElement.style.opacity = '0';
            return;
        }

        if (tooltip.body) {
            const titleLines = tooltip.title || [];
            const bodyLines = tooltip.body.map((i) => i.lines);

            let innerHtml = '';

            titleLines.forEach((t) => {
                innerHtml += `<div class="font-semibold">${t}</div>`;
            });

            bodyLines.forEach((b, i) => {
                // @ts-ignore
                if (tooltip.dataPoints[i] && tooltip.dataPoints[i].raw.type) {
                    // @ts-ignore
                    const background = i === 0 ? '#14b8a6' : '#f43f5e';
                    // @ts-ignore
                    const border = i === 0 ? '#5eead4' : '#fda4af';
                    innerHtml += `<div class="flex items-center gap-2 pt-2">`;
                    innerHtml += `<div class="w-4 h-4 rounded-full border-2" style="background:${background}; border-color:${border}"></div>`;
                    innerHtml += `<span>${b}</span>`;
                    innerHtml += `</div>`;
                }
            });

            // @ts-ignore
            const type = tooltip.dataPoints[0].raw.type;
            innerHtml += `<div class="flex items-center gap-2 pt-2">`;
            if (type === 'normal') {
                innerHtml += `<div class="flex h-5 w-5 rotate-45 flex-col items-center justify-center border-2 border-gray-300 bg-gray-500 text-xs font-bold text-gray-800">`;
            }
            if (type === 'risk') {
                innerHtml += `<div class="flex h-5 w-5 rotate-45 flex-col items-center justify-center border-2 border-red-300 bg-red-500 text-xs font-bold text-red-800">`;
            }
            if (type === 'decision') {
                innerHtml += `<div class="flex h-5 w-5 rotate-45 flex-col items-center justify-center border-2 border-lime-300 bg-lime-500 text-xs font-bold text-lime-800">`;
            }
            if (type === 'constraint') {
                innerHtml += `<div class="flex h-5 w-5 rotate-45 flex-col items-center justify-center border-2 border-amber-300 bg-amber-500 text-xs font-bold text-amber-800">`;
            }
            if (type === 'test') {
                innerHtml += `<div class="flex h-5 w-5 rotate-45 flex-col items-center justify-center border-2 border-blue-300 bg-blue-500 text-xs font-bold text-blue-800">`;
            }
            // @ts-ignore
            innerHtml += `<span class="-rotate-45 uppercase">${type.charAt(0)}</span></div>`;
            // @ts-ignore
            innerHtml += `<span class="capitalize">${type}</span>`;
            innerHtml += `</div>`;

            // @ts-ignore
            if (tooltip.dataPoints[0].raw.comments) {
                innerHtml += `<div class="pt-2 font-semibold">Comments</div>`;
                // @ts-ignore
                innerHtml += `<div class="rounded bg-slate-100 p-2">${tooltip.dataPoints[0].raw.comments}</div>`;
            }

            if (
                // @ts-ignore
                tooltip.dataPoints[0].raw.attachments &&
                // @ts-ignore
                tooltip.dataPoints[0].raw.attachments.length
            ) {
                innerHtml += `<div class="pt-2 font-semibold">Attachments</div>`;
                // @ts-ignore
                tooltip.dataPoints[0].raw.attachments.forEach((a) => {
                    innerHtml += `<div class class="pt-1 max-w-[10rem] truncate">`;
                    innerHtml += `<a class="link" target="_blank" href="${a}">${a}</a>`;
                    innerHtml += `</div>`;
                });
            }

            tooltipElement.innerHTML = innerHtml;
        }

        const { offsetLeft: positionX, offsetTop: positionY } = chart.canvas;

        tooltipElement.style.opacity = '1';
        tooltipElement.style.left = positionX + tooltip.caretX + 'px';
        tooltipElement.style.top = positionY + tooltip.caretY + 'px';
    };

    return { tooltipHandler };
}
