import moment, { Moment } from 'moment';
import { computed, ref } from 'vue';

export function useTimeScale(datasetCallback: Function) {
    const timeScales = {
        week: { format: '[Week] W YYYY' },
        month: { format: 'MMM YYYY' },
        year: { format: 'YYYY' },
    };
    const timeScale = ref<'week' | 'month' | 'year'>('week');

    const selectedDate = ref<Moment>(moment().startOf(timeScale.value));

    const useCallback = () => {
        datasetCallback();
    };

    const setScale = (scale: 'week' | 'month' | 'year') => {
        selectedDate.value = moment(selectedDate.value).startOf(scale);
        timeScale.value = scale;
        useCallback();
    };

    const timeTravel = (action: 'add' | 'subtract') => {
        selectedDate.value = moment(selectedDate.value)[action](1, timeScale.value);
        useCallback();
    };

    const labels = computed(() => {
        let values = [];

        let max = 7;
        let unit: moment.unitOfTime.DurationConstructor = 'days';

        if (timeScale.value === 'month') {
            max = selectedDate.value.daysInMonth();
            unit = 'days';
        }

        if (timeScale.value === 'year') {
            max = 12;
            unit = 'months';
        }

        for (let i = 0; i < max; i++) {
            values[i] = moment(selectedDate.value).add(i, unit).format('YYYY-MM-DD');
        }

        return values;
    });

    return { labels, selectedDate, setScale, timeScale, timeScales, timeTravel };
}
