<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Auth\Events\Lockout;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Event;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class SignInTest extends TestCase
{
    use RefreshDatabase;

    protected function signIn($data = []): TestResponse
    {
        return $this->postJson(
            route('auth.signin'),
            $data,
            ['referer' => 'localhost'],
        );
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->signIn();

        $response->assertInvalid([
            'email' => __('validation.required', ['attribute' => 'email']),
            'password' => __('validation.required', ['attribute' => 'password']),
        ]);
    }

    public function testCredentialsAreInvalid(): void
    {
        $data = [
            'email' => fake()->email(),
            'password' => fake()->word(),
        ];

        $response = $this->signIn($data);

        $response->assertInvalid(['email' => __('auth.failed')]);

        $this->assertGuest();
    }

    public function testSignInThrottle(): void
    {
        Event::fake();

        $data = [
            'email' => fake()->email(),
            'password' => fake()->word(),
        ];

        for ($i = 0; $i < 5; $i++) {
            $this->signIn($data)->assertInvalid(['email']);
        }

        $this->signIn($data)->assertInvalid(['email']);

        $this->assertGuest();

        Event::assertDispatched(Lockout::class);
    }

    public function testUserSignedIn(): void
    {
        $user = User::factory()->create();

        $response = $this->signIn([
            'email' => $user->email,
            'password' => 'password',
        ]);

        $this->assertAuthenticated();

        $response->assertJson([
            'message' => 'Welcome back!',
            'data' => [
                'id' => $user->id,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'email' => $user->email,
                'email_verified_at' => $user->email_verified_at->toJson(),
                'role' => $user->role,
            ],
        ]);
    }
}
