<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ProjectUserIndexTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);
    }

    protected function getUsers($page = 1): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.users.index',
                    [
                        'project' => $this->project,
                        'page' => $page,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getUsers();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getUsers();

        $response->assertNotFound();
    }

    public function testUsersListed(): void
    {
        User::factory()->count(2)->create();

        $users = User::factory()->count(5)->create();

        $this->project->users()->attach($users, ['role' => 'user']);

        $response = $this->getUsers();

        $response->assertJson([
            'meta' => [
                'current_page' => 1,
                'total' => 5,
                'per_page' => 12,
            ],
        ]);
    }

    public function testUsersListedSecondPage(): void
    {
        $users = User::factory()->count(15)->create();

        $this->project->users()->attach($users, ['role' => 'user']);

        $response = $this->getUsers(2);

        $response->assertJson([
            'meta' => [
                'current_page' => 2,
                'total' => 15,
                'per_page' => 12,
            ],
        ]);
    }
}
