<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ProjectUpdateTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'owner']);
    }

    protected function updateProject($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->putJson(route('projects.update', ['project' => $this->project]), $data);
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->updateProject();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->updateProject();

        $response->assertNotFound();
    }

    public function testProjectExistButUserNotAdminOrOwner(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->updateProject();

        $response->assertForbidden();
    }

    public function testUnverifiedUserCannotUpdateProject(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->updateProject();

        $response->assertForbidden();
    }

    public function testUserCannotUpdateProject(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->updateProject();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->updateProject();

        $response->assertInvalid([
            'name' => __('validation.required', ['attribute' => 'name']),
        ]);
    }

    public function testProjectWasUpdated(): void
    {
        $name = fake()->catchPhrase();

        $response = $this->updateProject(['name' => $name]);

        $response->assertValid();

        $response->assertJson([
            'data' => [
                'id' => $this->project->id,
                'name' => $name,
                'created_at' => $response->json('data.created_at'),
            ],
        ]);

        $this->assertDatabaseHas('projects', [
            'id' => $this->project->id,
            'name' => $name,
        ]);
    }
}
