<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ItemRecordUpdateTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    protected ItemRecord $record;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);

        $this->record = ItemRecord::factory()->create(['driver_item_id' => $this->item]);
    }

    protected function updateRecord($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->putJson(
                route(
                    'projects.drivers.items.records.update',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                        'record' => $this->record,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherProject(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testRecordNotFound(): void
    {
        $this->record->delete();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testRecordExistButBelongsToAnotherProject(): void
    {
        $this->record = ItemRecord::factory()->create();

        $response = $this->updateRecord();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotUpdateRecord(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->updateRecord();

        $response->assertForbidden();
    }

    public function testUserCannotUpdateRecord(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->updateRecord();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->updateRecord();

        $response->assertInvalid([
            'date' => __('validation.required', ['attribute' => 'date']),
            'max' => __('validation.required', ['attribute' => 'max']),
            'type' => __('validation.required', ['attribute' => 'type']),
        ]);
    }

    public function testDateFormatIsValid(): void
    {
        $response = $this->updateRecord(['date' => Str::random(10)]);

        $response->assertInvalid([
            'date' => __('validation.date_format', ['attribute' => 'date', 'format' => 'Y-m-d']),
        ]);
    }

    public function testDateIsUnique(): void
    {
        $record = ItemRecord::factory()->create(['driver_item_id' => $this->item]);

        $response = $this->updateRecord(['date' => $record->date->format('Y-m-d')]);

        $response->assertInvalid([
            'date' => __('validation.unique', ['attribute' => 'date']),
        ]);
    }

    public function testMinOnlyRequiredWhenItemIsRange(): void
    {
        $this->item->update(['range' => true]);

        $response = $this->updateRecord();

        $response->assertInvalid([
            'min' => __('validation.required', ['attribute' => 'min']),
        ]);
    }

    public function testMinNotRequiredWhenItemIsNotRange(): void
    {
        $this->item->update(['range' => false]);

        $response = $this->updateRecord();

        $response->assertJsonMissingValidationErrors('min');
    }

    public function testMaxAndMinAreNumeric(): void
    {
        $this->item->update(['range' => true]);

        $response = $this->updateRecord([
            'max' => Str::random(10),
            'min' => Str::random(10),
        ]);

        $response->assertInvalid([
            'max' => __('validation.numeric', ['attribute' => 'max']),
            'min' => __('validation.numeric', ['attribute' => 'min']),
        ]);
    }

    public function testMinIsLessThanMax(): void
    {
        $this->item->update(['range' => true]);

        $response = $this->updateRecord([
            'max' => 10,
            'min' => 15,
        ]);

        $response->assertInvalid([
            'min' => __('validation.lt.numeric', ['attribute' => 'min', 'value' => 10]),
        ]);
    }

    public function testTypeIsValid(): void
    {
        $response = $this->updateRecord(['type' => Str::Random(10)]);

        $response->assertInvalid([
            'type' => __('validation.in', ['attribute' => 'type']),
        ]);
    }

    public function testAttachmentsIsArray(): void
    {
        $response = $this->updateRecord(['attachments' => Str::random(10)]);

        $response->assertInvalid([
            'attachments' => __('validation.array', ['attribute' => 'attachments']),
        ]);
    }

    public function testAttachmentItemIsString(): void
    {
        $response = $this->updateRecord(['attachments' => [10]]);

        $response->assertInvalid([
            'attachments.0' => __('validation.string', ['attribute' => 'attachments.0']),
        ]);
    }

    public function testRecordWasUpdatedWithSameDate(): void
    {
        $record = ItemRecord::factory()->make();

        $data = $record->toArray();
        $data['date'] = $record->date->format('Y-m-d');

        $response = $this->updateRecord($data);

        $response->assertValid();

        $this->assertDatabaseHas('item_records', [
            'id' => $this->record->id,
            'driver_item_id' => $this->item->id,
            'date' => $record->date->toDatetimeString(),
            'max' => $record->max,
            'min' => $record->min,
            'type' => $record->type,
            'comments' => $record->comments,
        ]);
    }

    public function testRecordWasUpdatedWithDifferentDate(): void
    {
        $record = ItemRecord::factory()->make(['date' => now()->addDay()->startOfDay()]);

        $data = $record->toArray();
        $data['date'] = $record->date->format('Y-m-d');

        $response = $this->updateRecord($data);

        $response->assertValid();

        $this->assertDatabaseHas('item_records', [
            'id' => $this->record->id,
            'driver_item_id' => $this->item->id,
            'date' => $record->date->toDatetimeString(),
            'max' => $record->max,
            'min' => $record->min,
            'type' => $record->type,
            'comments' => $record->comments,
        ]);
    }
}
