<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ItemRecordShowTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    protected ItemRecord $record;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);

        $this->record = ItemRecord::factory()->create(['driver_item_id' => $this->item]);
    }

    protected function getRecord(): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.drivers.items.records.show',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                        'record' => $this->record,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherProject(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testRecordNotFound(): void
    {
        $this->record->delete();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testRecordExistButBelongsToAnotherProject(): void
    {
        $this->record = ItemRecord::factory()->create();

        $response = $this->getRecord();

        $response->assertNotFound();
    }

    public function testRecordFound(): void
    {
        $response = $this->getRecord();

        $this->assertDatabaseHas('item_records', [
            'id' => $this->record->id,
            'driver_item_id' => $this->item->id,
            'date' => $this->record->date->toDatetimeString(),
            'max' => $this->record->max,
            'min' => $this->record->min,
            'comments' => $this->record->comments,
        ]);
    }
}
