<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ItemRecordIndexTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function getRecords(): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.drivers.items.records.index',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherProject(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testRecordsListed(): void
    {
        ItemRecord::factory()->count(2)->create();

        $records = ItemRecord::factory()
            ->count(5)
            ->state(new Sequence(
                ['date' => now()->startOfDay()],
                ['date' => now()->startOfDay()->addDays()],
                ['date' => now()->startOfDay()->addDays(2)],
                ['date' => now()->startOfDay()->addDays(3)],
                ['date' => now()->startOfDay()->addDays(4)],
            ))
            ->make();

        $this->item->records()->createMany($records->toArray());

        $response = $this->getRecords();

        $this->assertCount(5, $response->json('data'));
    }
}
