<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class DriverItemStoreTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);
    }

    protected function storeItem($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->postJson(
                route(
                    'projects.drivers.items.store',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->storeItem();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->storeItem();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->storeItem();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->storeItem();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotStoreItem(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->storeItem();

        $response->assertForbidden();
    }

    public function testUserCannotStoreItem(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->storeItem();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->storeItem();

        $response->assertInvalid([
            'name' => __('validation.required', ['attribute' => 'name']),
            'range' => __('validation.required', ['attribute' => 'range']),
        ]);
    }

    public function testNameIsUnique(): void
    {
        $item = DriverItem::factory()->create(['key_driver_id' => $this->driver->id]);

        $response = $this->storeItem(['name' => $item->name]);

        $response->assertInvalid([
            'name' => __('validation.unique', ['attribute' => 'name']),
        ]);
    }

    public function testRangeIsBoolean(): void
    {
        $response = $this->storeItem(['range' => Str::random(10)]);

        $response->assertInvalid([
            'range' => __('validation.boolean', ['attribute' => 'range']),
        ]);
    }

    public function testMaxAndMinAreNumeric(): void
    {
        $response = $this->storeItem([
            'max' => Str::random(10),
            'min' => Str::random(10),
        ]);

        $response->assertInvalid([
            'max' => __('validation.numeric', ['attribute' => 'max']),
            'min' => __('validation.numeric', ['attribute' => 'min']),
        ]);
    }

    public function testMinIsLessThanMax(): void
    {
        $response = $this->storeItem([
            'max' => 10,
            'min' => 15,
        ]);

        $response->assertInvalid([
            'min' => __('validation.lt.numeric', ['attribute' => 'min', 'value' => 10]),
        ]);
    }

    public function testItemWasSaved(): void
    {
        $item = DriverItem::factory()->make();

        $response = $this->storeItem($item->only(['name', 'range', 'max', 'min']));

        $response->assertValid();

        $this->assertDatabaseHas('driver_items', [
            'id' => $response->json('data.id'),
            'key_driver_id' => $this->driver->id,
            'name' => $item->name,
            'range' => $item->range,
            'max' => $item->max,
            'min' => $item->min,
        ]);
    }
}
