<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Auth\Notifications\VerifyEmail;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class AccountUpdateTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
    }

    protected function updateProfile($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->putJson(route('account.update'), $data);
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->updateProfile();

        $response->assertInvalid([
            'first_name' => __('validation.required', ['attribute' => 'first name']),
            'last_name' => __('validation.required', ['attribute' => 'last name']),
            'email' => __('validation.required', ['attribute' => 'email']),
        ]);
    }

    public function testEmailMustBeValid(): void
    {
        $response = $this->updateProfile(['email' => Str::random(10)]);

        $response->assertInvalid(['email' => __('validation.email', ['attribute' => 'email'])]);
    }

    public function testEmailMustBeUnique(): void
    {
        $user = User::factory()->create();

        $response = $this->updateProfile(['email' => $user->email]);

        $response->assertInvalid(['email' => __('validation.unique', ['attribute' => 'email'])]);
    }

    public function testProfileUpdatedWithSameEmail(): void
    {
        Notification::fake();

        $data = [
            'first_name' => fake()->firstName(),
            'last_name' => fake()->lastName(),
            'email' => $this->user->email,
        ];

        $response = $this->updateProfile($data);

        $response->assertJson(['message' => 'Your information was updated successfully.']);

        $this->assertDatabaseHas('users', [
            'id' => $this->user->id,
            ...$data,
        ]);

        Notification::assertNothingSent();
    }

    public function testProfileUpdatedWithDifferentEmail(): void
    {
        Notification::fake();

        $data = [
            'first_name' => fake()->firstName(),
            'last_name' => fake()->lastName(),
            'email' => fake()->email(),
        ];

        $response = $this->updateProfile($data);

        $response->assertJson(['message' => 'Your information was updated successfully.']);

        $this->assertDatabaseHas('users', [
            'id' => $this->user->id,
            ...$data,
        ]);

        Notification::assertSentTo($this->user, VerifyEmail::class);
    }
}
