<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class AccountPasswordTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
    }

    protected function updatePassword($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->putJson(route('account.password'), $data);
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->updatePassword();

        $response->assertInvalid([
            'current_password' => __('validation.required', ['attribute' => 'current password']),
            'password' => __('validation.required', ['attribute' => 'password']),
        ]);
    }

    public function testCurrentPasswordInvalid(): void
    {
        $response = $this->updatePassword(['current_password' => Str::random(10)]);

        $response->assertInvalid([
            'current_password' => __('validation.current_password', ['attribute' => 'current password']),
        ]);
    }

    public function testPasswordNotConfirmed(): void
    {
        $response = $this->updatePassword(['password' => Str::random(10)]);

        $response->assertInvalid([
            'password' => __('validation.confirmed', ['attribute' => 'password']),
        ]);
    }

    public function testPasswordTooShort(): void
    {
        $response = $this->updatePassword(['password' => Str::random(7)]);

        $response->assertInvalid([
            'password' => __('validation.min.string', ['attribute' => 'password', 'min' => 8]),
        ]);
    }

    public function testPasswordSaved(): void
    {
        $newPassword = Str::random(10);

        $response = $this->updatePassword([
            'current_password' => 'password',
            'password' => $newPassword,
            'password_confirmation' => $newPassword,
        ]);

        $response->assertJson(['message' => 'Your new password was saved succesfully.']);

        $this->assertTrue(Hash::check($newPassword, $this->user->fresh()->password));
    }
}
