<?php

namespace App\Http\Requests;

use Illuminate\Database\Query\Builder;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\File;
use Illuminate\Validation\ValidationException;

class ItemRecordsImport extends FormRequest
{
    public function authorize(): bool
    {
        $user = $this->project->users()->find($this->user()->id);

        return $user->hasVerifiedEmail() && ($user->pivot->role === 'owner' || $user->pivot->role === 'admin');
    }

    public function rules(): array
    {
        return [
            'file' => [
                'required',
                File::types(['csv', 'txt']),
            ],
        ];
    }

    public function saveData(): int
    {
        $data = str_getcsv($this->file('file')->getContent(), "\n");

        $mappedData = [];

        for ($i = 1; $i < count($data); $i++) {
            $fields = explode(',', $data[$i]);

            $model = [
                'date' => $fields[0],
                'max' => $fields[1],
                'min' => $fields[2] === '' ? null : $fields[2],
                'type' => $fields[3],
                'comments' => $fields[4],
            ];

            $mappedData[] = $this->validateRecord($model, $i + 1);
        }

        $uniqueDates = array_unique(collect($mappedData)->map->date->toArray());

        if (count($uniqueDates) !== count($mappedData)) {
            throw ValidationException::withMessages(['date' => 'The date field is not unique within the file.']);
        }

        $records = $this->item->records()->createMany($mappedData);

        return $records->count();
    }

    protected function validateRecord(array $model, int $row): array
    {
        $validator = Validator::make(
            $model, [
                'date' => [
                    'required',
                    'date_format:Y-m-d',
                    Rule::unique('item_records')->where(function (Builder $query) {
                        $query->where('driver_item_id', $this->item->id);
                    }),
                ],
                'max' => ['required', 'numeric'],
                'min' => [
                    $this->item->range ? 'required' : 'nullable',
                    'numeric',
                    'lt:max',
                ],
                'type' => [
                    'required',
                    Rule::in(['normal', 'decision', 'risk', 'constraint', 'test']),
                ],
                'comments' => ['nullable', 'string', 'max:1000'],
            ],
            [
                '*.required' => 'The field :attribute is required in line '.$row.'.',
                '*.date_format' => 'The :attribute field must match the format :format in line '.$row.'.',
                '*.unique' => 'The :attribute has already been taken in line '.$row.'.',
                '*.numeric' => 'The :attribute field must be a number in line '.$row.'.',
                '*.lt' => 'The :attribute field must be less than :value in line '.$row.'.',
                '*.in' => 'The selected :attribute is invalid in line '.$row.'.',
                '*.max' => 'The :attribute field must not be greater than :max characters in line '.$row.'.',
            ],
        );

        if ($validator->fails()) {
            throw ValidationException::withMessages($validator->errors()->toArray());
        }

        return $validator->validated();
    }
}
