<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProjectUserStore;
use App\Http\Resources\UserResource;
use App\Models\Project;
use App\Models\User;
use App\Notifications\ProjectUserRemoved;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Auth;

class ProjectUserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Project $project, Request $request): AnonymousResourceCollection
    {
        $users = $project->users()
            ->where('id', '<>', Auth::id())
            ->whereSearch($request->get('search'))
            ->orderByName()
            ->paginate(12);

        return UserResource::collection($users);
    }

    public function store(Project $project, ProjectUserStore $request): UserResource
    {
        $user = $request->addUser();

        return (new UserResource($user))
            ->additional(['message' => 'The user was added as collaborator to the project.']);
    }

    public function delete(Project $project, User $user): UserResource
    {
        $authUser = $project->users()->find(Auth::id());

        abort_if(
            ! $authUser->hasVerifiedEmail() ||
            $authUser->pivot->role === 'user' ||
            $user->pivot->role === 'owner' ||
            ($authUser->pivot->role === 'admin' && $user->pivot->role === 'admin'),
            403,
            'This action is unauthorized.',
        );

        $project->users()->detach($user);

        $user->notify(new ProjectUserRemoved($project));

        return (new UserResource($user))
            ->additional(['message' => 'The user was removed from the project.']);
    }
}
