<?php

namespace App\Http\Controllers;

use App\Http\Requests\KeyDriverStore;
use App\Http\Requests\KeyDriverUpdate;
use App\Http\Resources\KeyDriverResource;
use App\Models\KeyDriver;
use App\Models\Project;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Auth;

class KeyDriverController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Project $project): AnonymousResourceCollection
    {
        $drivers = $project->drivers()->with('items')->get();

        return KeyDriverResource::collection($drivers);
    }

    public function show(Project $project, KeyDriver $driver): KeyDriverResource
    {
        return new KeyDriverResource($driver);
    }

    public function store(Project $project, KeyDriverStore $request): KeyDriverResource
    {
        $driver = $request->storeDriver();

        return (new KeyDriverResource($driver))
            ->additional(['message' => 'The key driver was saved successfully.']);
    }

    public function update(Project $project, KeyDriver $driver, KeyDriverUpdate $request): KeyDriverResource
    {
        $request->updateDriver();

        return (new KeyDriverResource($driver))
            ->additional(['message' => 'The key driver was updated successfully.']);
    }

    public function delete(Project $project, KeyDriver $driver): KeyDriverResource
    {
        $user = $project->users()->find(Auth::id());

        abort_if(
            ! $user->hasVerifiedEmail() ||
            $user->pivot->role === 'user',
            403,
            'This action is unauthorized.'
        );

        $driver->delete();

        return (new KeyDriverResource($driver))
            ->additional(['message' => 'The key driver was removed successfully.']);
    }
}
