<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class UserIndexTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
    }

    protected function getUsers($params = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(route('users.index', $params));
    }

    public function testUsersListed(): void
    {
        User::factory()->count(5)->create();

        $response = $this->getUsers();

        $response->assertJson([
            'meta' => [
                'current_page' => 1,
                'total' => 5,
                'per_page' => 12,
            ],
        ]);
    }

    public function testUsersListedSecondPage(): void
    {
        User::factory()->count(15)->create();

        $response = $this->getUsers(['page' => 2]);

        $response->assertJson([
            'meta' => [
                'current_page' => 2,
                'total' => 15,
                'per_page' => 12,
            ],
        ]);
    }

    public function testUserPerPageInvalid(): void
    {
        User::factory()->count(10)->create();

        $response = $this->getUsers(['per_page' => fake()->word()]);

        $response->assertJson([
            'meta' => [
                'current_page' => 1,
                'from' => 1,
                'to' => 10,
                'total' => 10,
                'per_page' => 12,
            ],
        ]);
    }

    public function testUsersListedPerPage(): void
    {
        User::factory()->count(10)->create();

        $response = $this->getUsers(['per_page' => 5]);

        $response->assertJson([
            'meta' => [
                'current_page' => 1,
                'from' => 1,
                'to' => 5,
                'total' => 10,
                'per_page' => 5,
            ],
        ]);
    }

    public function testUsersFiltered(): void
    {
        $term = fake()->word();

        User::factory()->create(['first_name' => fake()->firstName().$term]);
        User::factory()->create(['first_name' => $term.fake()->firstName()]);

        User::factory()->create(['last_name' => fake()->lastName().$term]);
        User::factory()->create(['last_name' => $term.fake()->lastName()]);

        User::factory()->create(['email' => fake()->email().$term]);
        User::factory()->create(['email' => $term.fake()->email()]);

        User::factory()->count(3)->create();

        $response = $this->getUsers(['search' => $term]);

        $response->assertJson([
            'meta' => [
                'current_page' => 1,
                'total' => 6,
                'per_page' => 12,
            ],
        ]);
    }

    public function testUsersThatNotBelongToProject(): void
    {
        User::factory()->count(4)->create();

        $user = User::factory()->create();

        $project = Project::factory()->create();

        $this->user->projects()->attach($project, ['role' => 'user']);

        $user->projects()->attach($project, ['role' => 'user']);

        $response = $this->getUsers(['project_id' => $project->id]);

        $response->assertJson([
            'meta' => [
                'current_page' => 1,
                'total' => 4,
            ],
        ]);
    }
}
