<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Auth\Notifications\VerifyEmail;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class SignUpTest extends TestCase
{
    use RefreshDatabase;

    protected function signUp($data = []): TestResponse
    {
        return $this->postJson(route('auth.signup'), $data);
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->signUp();

        $response->assertInvalid([
            'first_name' => __('validation.required', ['attribute' => 'first name']),
            'last_name' => __('validation.required', ['attribute' => 'last name']),
            'email' => __('validation.required', ['attribute' => 'email']),
            'password' => __('validation.required', ['attribute' => 'password']),
        ]);
    }

    public function testEmailMustBeValid(): void
    {
        $response = $this->signUp(['email' => Str::random(10)]);

        $response->assertInvalid(['email' => __('validation.email', ['attribute' => 'email'])]);
    }

    public function testEmailMustBeUnique(): void
    {
        $user = User::factory()->create();

        $response = $this->signUp(['email' => $user->email]);

        $response->assertInvalid(['email' => __('validation.unique', ['attribute' => 'email'])]);
    }

    public function testPasswordNotConfirmed(): void
    {
        $response = $this->signUp(['password' => Str::random(10)]);

        $response->assertInvalid([
            'password' => __('validation.confirmed', ['attribute' => 'password']),
        ]);
    }

    public function testPasswordTooShort(): void
    {
        $response = $this->signUp(['password' => Str::random(7)]);

        $response->assertInvalid([
            'password' => __('validation.min.string', ['attribute' => 'password', 'min' => 8]),
        ]);
    }

    public function testUserRegistered(): void
    {
        Notification::fake();

        $data = [
            'first_name' => fake()->firstName(),
            'last_name' => fake()->lastName(),
            'email' => fake()->email(),
            'password' => Str::random(10),
        ];

        $data['password_confirmation'] = $data['password'];

        $response = $this->signUp($data);

        $this->assertDatabaseHas('users', [
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'email' => $data['email'],
            'email_verified_at' => null,
        ]);

        $this->assertAuthenticated();

        $user = User::where('email', $data['email'])->first();

        Notification::assertSentTo($user, VerifyEmail::class);

        $response->assertJson([
            'data' => [
                'id' => $user->id,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'email' => $user->email,
                'email_verified_at' => null,
                'role' => $user->role,
            ],
        ]);
    }
}
