<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\User;
use App\Notifications\ProjectUserAddded;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ProjectUserStoreTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);
    }

    protected function storeUser($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->postJson(
                route(
                    'projects.users.store',
                    [
                        'project' => $this->project,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->storeUser();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->storeUser();

        $response->assertNotFound();
    }

    public function testProjectExistButUserNotAdminOrOwner(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->storeUser();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->storeUser();

        $response->assertInvalid([
            'user_id' => __('validation.required', ['attribute' => 'user id']),
            'role' => __('validation.required', ['attribute' => 'role']),
        ]);
    }

    public function testUserExist(): void
    {
        $response = $this->storeUser(['user_id' => 1000]);

        $response->assertInvalid([
            'user_id' => __('validation.exists', ['attribute' => 'user id']),
        ]);
    }

    public function testRoleIsValid(): void
    {
        $response = $this->storeUser(['role' => Str::random(10)]);

        $response->assertInvalid([
            'role' => __('validation.in', ['attribute' => 'role']),
        ]);
    }

    public function testUserWasAddedToProject(): void
    {
        Notification::fake();

        $user = User::factory()->create();

        $response = $this->storeUser([
            'user_id' => $user->id,
            'role' => 'user',
        ]);

        $response->assertValid();

        $this->assertDatabaseHas('project_user', [
            'project_id' => $this->project->id,
            'user_id' => $user->id,
            'role' => 'user',
        ]);

        Notification::assertSentTo($user, ProjectUserAddded::class);
    }
}
