<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\User;
use App\Notifications\ProjectUserRemoved;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ProjectUserDeleteTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected User $collaborator;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->collaborator = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->collaborator->projects()->attach($this->project, ['role' => 'user']);
    }

    protected function removeUser(): TestResponse
    {
        return $this->actingAs($this->user)
            ->deleteJson(
                route(
                    'projects.users.delete',
                    [
                        'project' => $this->project,
                        'user' => $this->collaborator,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->removeUser();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->removeUser();

        $response->assertNotFound();
    }

    public function testProjectExistButUserNotAdminOrOwner(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testUserNotFound(): void
    {
        $this->collaborator->delete();

        $response = $this->removeUser();

        $response->assertNotFound();
    }

    public function testUserExistsButIsNotACollaborator(): void
    {
        $this->project->users()->detach($this->collaborator);

        $response = $this->removeUser();

        $response->assertNotFound();
    }

    public function testUserCannotDeleteOwner(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $this->collaborator->projects()->updateExistingPivot($this->project, ['role' => 'owner']);

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testAdminCannotDeleteOwner(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'admin']);

        $this->collaborator->projects()->updateExistingPivot($this->project, ['role' => 'owner']);

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testUserCannotDeleteAdmin(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $this->collaborator->projects()->updateExistingPivot($this->project, ['role' => 'admin']);

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testAdminCannotDeleteAdmin(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'admin']);

        $this->collaborator->projects()->updateExistingPivot($this->project, ['role' => 'admin']);

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testAdminCannotDeleteHimself(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'admin']);

        $this->collaborator = $this->user;

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testOwnerCannotDeleteHimself(): void
    {
        $this->collaborator = $this->user;

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testUserCannotDeleteUser(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->removeUser();

        $response->assertForbidden();
    }

    public function testUserWasRemovedFromProject(): void
    {
        Notification::fake();

        $response = $this->removeUser();

        $response->assertValid();

        $this->assertDatabaseMissing('project_user', [
            'project_id' => $this->project->id,
            'user_id' => $this->collaborator->id,
        ]);

        Notification::assertSentTo($this->collaborator, ProjectUserRemoved::class);
    }
}
